#  _________________________________________________________________________
#
#  Coopr: A COmmon Optimization Python Repository
#  Copyright (c) 2008 Sandia Corporation.
#  This software is distributed under the BSD License.
#  Under the terms of Contract DE-AC04-94AL85000 with Sandia Corporation,
#  the U.S. Government retains certain rights in this software.
#  For more information, see the Coopr README.txt file.
#  _________________________________________________________________________


import re
from coopr.opt.base import *
from coopr.opt.results import *
from coopr.opt.solver import *
import pyutilib.services
import pyutilib.common
import pyutilib.common
import pyutilib.plugin.core
import mockmip
import os
import copy


class PICO(SystemCallSolver):
    """The PICO LP/MIP solver
    """

    def __init__(self, **kwds):
        #
        # Call base constructor
        #
        kwds["type"] = "pico"
        SystemCallSolver.__init__(self, **kwds)
        #
        # Setup valid problem formats, and valid results for each problem format
        #
        self._valid_problem_formats=[ProblemFormat.cpxlp, ProblemFormat.nl, ProblemFormat.mps]
        self._valid_result_formats = {}
        self._valid_result_formats[ProblemFormat.cpxlp] = [ResultsFormat.soln]
        self._valid_result_formats[ProblemFormat.nl] = [ResultsFormat.sol]
        self._valid_result_formats[ProblemFormat.mps] = [ResultsFormat.soln]

    def executable(self):
        executable = pyutilib.services.registered_executable("PICO")
        if executable is None:
            pyutilib.plugin.core.PluginGlobals.env().log.error("Could not locate the 'PICO' executable, which is required for solver %s" % self.name)
            self.enable = False
            return None
        return executable.get_path()

    def create_command_line(self,executable,problem_files):
        #
        # Define log file
        #
        if self.log_file is None:
           self.log_file = pyutilib.services.TempfileManager.create_tempfile(suffix="PICO.log")
        fname = problem_files[0].split(os.sep)[-1]
        if '.' in fname:
            tmp = fname.split('.')
            if len(tmp) > 2:
                fname = '.'.join(tmp[:-1])
            else:
                fname = tmp[0]
        if self.soln_file is None:
           self.soln_file = pyutilib.services.TempfileManager.create_tempfile(suffix=fname+".soln")
        self.sol_file = fname+".sol"
        #
        # Define results file
        #
        if self._results_format is None or self._results_format == ResultsFormat.soln:
           self.results_file = self.soln_file
        elif self._results_format == ResultsFormat.sol:
           self.results_file = self.sol_file
        #  
        # Eventually, these formats will be added to PICO...
        #
        #elif self._results_format == ResultsFormat.osrl:
           #self.results_file = self.tmpDir+os.sep+"PICO.osrl.xml"
        #
        # Define command line
        #
        if (self.mipgap is not None):
           raise ValueError, "The mipgap parameter is currently not being processed by PICO solver plugin"        
        env=copy.copy(os.environ)
        if self._problem_format is None or self._problem_format == ProblemFormat.nl:
           if "debug" in self.options:
                opts = []
           else:
                opts = ["debug=2"]
           for key in self.options:
                if isinstance(self.options[key],basestring) and ' ' in self.options[key]:
                    opts.append(key+"=\""+str(self.options[key])+"\"")
                else:
                    opt.append(key+"="+str(self.options[key]))
           env["PICO_options"] = " ".join(opts)
           proc = self._timer + " " + executable + " " + problem_files[0] + " -AMPL"
        elif self._problem_format == ProblemFormat.cpxlp or self._problem_format == ProblemFormat.mps:
           if "debug" in self.options:
                opt = ""
           else:
                opt = " --debug 2"
           for key in self.options:
                if isinstance(self.options[key],basestring) and ' ' in self.options[key]:
                    opt += " --"+key+"=\""+str(self.options[key])+"\""
                else:
                    opt += " --"+key+"="+str(self.options[key])
           proc = self._timer + " " + executable + opt + " --output " + self.soln_file + " " + problem_files[0]
        return pyutilib.misc.Bunch(cmd=proc, log_file=self.log_file, env=env)

    def process_logfile(self):
        """
        Process a logfile
        """
        results = SolverResults()
        #
        # Initial values
        #
        #results.solver.statistics.branch_and_bound.number_of_created_subproblems=0
        #results.solver.statistics.branch_and_bound.number_of_bounded_subproblems=0
        soln = results.solution.add()
        soln.objective['f'].value = None
        #
        # Process logfile
        #
        OUTPUT = open(self.log_file)
        output = "".join(OUTPUT.readlines())
        OUTPUT.close()
        #
        # Parse logfile lines
        #
        for line in output.split("\n"):
          tokens = re.split('[ \t]+',line.strip())
          if len(tokens) > 3 and tokens[0] == "ABORTED:":
             results.solver.status=SolverStatus.aborted
          elif len(tokens) == 5 and tokens[0] == "Final" and tokens[1] == "Solution:":
             soln.objective['f'].value = tokens[4]
             soln.status = SolutionStatus.optimal
          elif len(tokens) == 3 and tokens[0] == "LP" and tokens[1] == "value=":
             soln.objective['f'].value = tokens[2]
             soln.status=SolutionStatus.optimal
             if results.problem.sense == ProblemSense.minimize:
                results.problem.lower_bound = tokens[2]
             else:
                results.problem.upper_bound = tokens[2]
          elif len(tokens) == 2 and tokens[0] == "Bound:":
             if results.problem.sense == ProblemSense.minimize:
                results.problem.lower_bound = tokens[1]
             else:
                results.problem.upper_bound = tokens[1]
          elif len(tokens) == 3 and tokens[0] == "Created":
             results.solver.statistics.branch_and_bound.number_of_created_subproblems = tokens[1]
          elif len(tokens) == 3 and tokens[0] == "Bounded":
             results.solver.statistics.branch_and_bound.number_of_bounded_subproblems = tokens[1]
          elif len(tokens) == 2 and tokens[0] == "sys":
             results.solver.system_time=tokens[1]
          elif len(tokens) == 2 and tokens[0] == "user":
             results.solver.user_time=tokens[1]
          elif len(tokens) == 3 and tokens[0] == "Solving" and tokens[1] == "problem:":
             results.problem.name = tokens[2]
          elif len(tokens) == 4 and tokens[2] == "constraints:":
             results.problem.number_of_constraints = tokens[3]
          elif len(tokens) == 4 and tokens[2] == "variables:":
             results.problem.number_of_variables = tokens[3]
          elif len(tokens) == 4 and tokens[2] == "nonzeros:":
             results.problem.number_of_nonzeros = tokens[3]
          elif len(tokens) == 3 and tokens[1] == "Sense:":
             if tokens[2] == "minimization":
                results.problem.sense = ProblemSense.minimize
             else:
                results.problem.sense = ProblemSense.maximize

        if results.solver.status is SolverStatus.aborted:
           soln.optimality=SolutionStatus.unsure
        if soln.status is SolutionStatus.optimal:
           soln.gap=0.0
           if results.problem.sense == ProblemSense.minimize:
                results.problem.lower_bound = soln.objective['f'].value
                if "upper_bound" in dir(results.problem):
                    del results.problem.upper_bound
           else:
                results.problem.upper_bound = soln.objective['f'].value
                if "lower_bound" in dir(results.problem):
                    del results.problem.lower_bound
        if results.solver.status is SolverStatus.error:
           results.solution.delete(0)
        return results

    def process_soln_file(self,results):
        #print "HERE",self._results_format, self._results_format is ResultsFormat.sol, self.results_reader
        if self._results_format is ResultsFormat.sol:
           return

        #if os.path.exists(self.sol_file):
            #results_reader = ReaderFactory(ResultsFormat.sol)
            #results = results_reader(self.sol_file, results, results.solution(0))
            #return

        if not os.path.exists(self.soln_file):
           return
        soln = results.solution(0)
        results.problem.num_objectives=1
        tmp=[]
        flag=False
        INPUT = open(self.soln_file,"r")
        lp_flag=None
        for line in INPUT:
            tokens = re.split('[ \t]+',line.strip())
            if len(tokens) == 0 or (len(tokens) == 1 and tokens[0]==''):
                continue
            if tokens[0] == "Objective":
                continue
            #print "LINE",tokens
            if lp_flag is None:
                lp_flag = (tokens[0] == "LP")
                continue
            if tokens[0] == "Dual" and tokens[1] == "solution:":
                # It looks like we've just been processing primal
                # variables.
                for (var,val) in tmp:
                    soln.variable[var] = val
                tmp=[]
                continue
            if len(tokens) < 3:
                print "ERROR", line,tokens
            tmp.append( (tokens[0],eval(tokens[2])) )
        if lp_flag:
            for (var,val) in tmp:
                soln.constraint[var].dual = val
        else:
            for (var,val) in tmp:
                soln.variable[var] = val 
        INPUT.close()


class MockPICO(PICO,mockmip.MockMIP):
    """A Mock PICO solver used for testing
    """

    def __init__(self, **kwds):
        try:
           PICO.__init__(self,**kwds)
        except pyutilib.common.ApplicationError: #pragma:nocover
           pass                        #pragma:nocover
        mockmip.MockMIP.__init__(self,"pico")

    def available(self, exception_flag=True):
        return PICO.available(self,exception_flag)

    def create_command_line(self,executable,problem_files):
        command = PICO.create_command_line(self,executable,problem_files)
        mockmip.MockMIP.create_command_line(self,executable,problem_files)
        return command

    def executable(self):
        return mockmip.MockMIP.executable(self)

    def _execute_command(self,cmd):
        return mockmip.MockMIP._execute_command(self,cmd)


pyutilib.services.register_executable(name="pico")
SolverRegistration("pico", PICO)
SolverRegistration("_mock_pico", MockPICO)
