# copyright 2003-2011 LOGILAB S.A. (Paris, FRANCE), all rights reserved.
# contact http://www.logilab.fr/ -- mailto:contact@logilab.fr
#
# This file is part of CubicWeb.
#
# CubicWeb is free software: you can redistribute it and/or modify it under the
# terms of the GNU Lesser General Public License as published by the Free
# Software Foundation, either version 2.1 of the License, or (at your option)
# any later version.
#
# CubicWeb is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more
# details.
#
# You should have received a copy of the GNU Lesser General Public License along
# with CubicWeb.  If not, see <http://www.gnu.org/licenses/>.
"""Basic views for python values (eg without any result set)
"""
__docformat__ = "restructuredtext en"

from cubicweb.view import View
from cubicweb.selectors import match_kwargs
from cubicweb.web.views import tableview


class PyValTableColRenderer(tableview.AbstractColumnRenderer):
    """Default column renderer for :class:`PyValTableView`."""
    def bind(self, view, colid):
        super(PyValTableColRenderer, self).bind(view, colid)
        self.header = view.headers[colid] if view.headers else None
        self.data = view.pyvalue

    def render_header(self, w):
        if self.header:
            w(self._cw._(self.header))
        else:
            w(self.empty_cell_content)

    def render_cell(self, w, rownum):
        w(unicode(self.data[rownum][self.colid]))


class PyValTableView(tableview.TableMixIn, View):
    """This table view is designed to be used a list of list of unicode values
    given as a mandatory `pyvalue` argument. Take care, content is NOT
    xml-escaped.

    It's configured through the following selection arguments.

    If `headers` is specified, it is expected to be a list of headers to be
    inserted as first row (in <thead>).

    `header_column_idx` may be used to specify a column index or a set of column
    indiced where values should be inserted inside <th> tag instead of <td>.

    `cssclass` is the CSS class used on the <table> tag, and default to
    'listing' (so that the table will look similar to those generated by the
    table view).
    """
    __regid__ = 'pyvaltable'
    __select__ = match_kwargs('pyvalue')
    default_column_renderer_class = PyValTableColRenderer
    paginable = False # not supported
    headers = None
    cssclass = None
    domid = None

    def __init__(self, req, pyvalue, headers=None, cssclass=None,
                 header_column_idx=None, **kwargs):
        super(PyValTableView, self).__init__(req, **kwargs)
        self.pyvalue = pyvalue
        if headers is not None:
            self.headers = headers
        elif self.headers: # headers set on a class attribute, translate
            self.headers = [self._cw._(header) for header in self.headers]
        if cssclass is not None:
            self.cssclass = cssclass
        self.header_column_idx = header_column_idx

    @property
    def layout_args(self):
        args = {}
        if self.cssclass:
            args['cssclass'] = self.cssclass
        if self.header_column_idx is not None:
            args['header_column_idx'] = self.header_column_idx
        return args

    # layout callbacks #########################################################

    @property
    def table_size(self):
        """return the number of rows (header excluded) to be displayed"""
        return len(self.pyvalue)

    @property
    def has_headers(self):
        return self.headers

    def build_column_renderers(self):
        return [self.column_renderer(colid)
                for colid in xrange(len(self.pyvalue[0]))]

    def facets_form(self, mainvar=None):
        return None # not supported

    def table_actions(self):
        return [] # not supported


class PyValListView(View):
    """display a list of values into an html list.

    Take care, content is NOT xml-escaped.
    """
    __regid__ = 'pyvallist'
    __select__ = match_kwargs('pyvalue')

    def call(self, pyvalue):
        self.w(u'<ul>\n')
        for line in pyvalue:
            self.w(u'<li>%s</li>\n' % line)
        self.w(u'</ul>\n')
