import click
from dailybible.lib.reader import DailyBibleReader

VERSION = '1.0.3'
reader = DailyBibleReader()

@click.group(invoke_without_command=True)
@click.option('--version', '-v', is_flag=True, help='Display application\
                                                     version.')
@click.pass_context
def cli(ctx, version):
    """Read the Bible in a year."""
    if version:
        click.echo('dailybible %s' % (VERSION))
        return
    if ctx.invoked_subcommand is None:
        read()

@cli.command('read', short_help='read today\'s passages (default)')
@click.option('--browser', '-b', is_flag=True, help='Read passages in a web \
                                                     web browser.')
def read(browser):
    """Display today's Bible passages."""
    if browser:
        reader.set_output_format('html')
    reader.read_passage()

@cli.command('next', short_help='read tomorrow\'s passages')
@click.option('--browser', '-b', is_flag=True, help='Read passages in a web \
                                                     web browser.')
def next(browser):
    """Increments reading progress and displays tomorrow's Bible
    passages.
    Subsequent read commands will be affected.
    """
    if browser:
        reader.set_output_format('html')
    reader.read_next_passage()

@cli.command('previous', short_help='read yesterday\'s passages')
@click.option('--browser', '-b', is_flag=True, help='Read passages in a web \
                                                     web browser.')
def previous(browser):
    """Decrements reading progress and displays yesterday's Bible
    passages.
    Subsequent read commands will be affected.
    """
    if browser:
        reader.set_output_format('html')
    reader.read_previous_passage()

@cli.command('status', short_help='reading plan progress info')
def status():
    """Displays information on reading plan progress."""
    messages = reader.get_status_messages()
    for message in messages:
        click.echo(message)

@cli.command('reset', short_help='reset reading progress')
def reset():
    """Resets reading plan data by setting the starting date to today
    and progress to zero.
    """
    if click.confirm('Reset your reading progress?'):
        reader.config.set_start_date()
        reader.config.set_progress(0)
        click.echo('Reading progress has been reset.')

