import os
import click
import configparser
from datetime import date, datetime, timedelta

APP_NAME = 'dailybible'

class AppConfig(object):
    def __init__(self):
        """Creates an ini file to store starting date and progress
        data for the application.
        """
        config_dir = click.get_app_dir(APP_NAME)
        self.cfg_path = os.path.join(config_dir, 'config.ini')
        self.config = configparser.ConfigParser()

        # check if app config directory exists; if not, create it
        if not os.path.exists(config_dir):
            os.makedirs(config_dir)

        # check if app config file exists; if not, create it
        if not os.path.isfile(self.cfg_path):
            now = date.today().strftime('%Y/%m/%d')
            self.config['DEFAULT'] = {
                'StartDate': now,
                'Progress' : 0
            }
            self.save_config()
        else:
            self.config.read(self.cfg_path)

    def save_config(self):
        """Saves data file to application config file."""
        with click.open_file(self.cfg_path, 'w+') as configfile:
            self.config.write(configfile)

    def get_start_date(self):
        """Get 'StartDate' from config and return as date object."""
        value = self.config['DEFAULT']['StartDate']
        return datetime.strptime(value, '%Y/%m/%d')

    def get_start_datestr(self):
        """Get 'StartDate' string from config"""
        return self.config['DEFAULT']['StartDate']

    def set_start_date(self, dateobj=date.today()):
        """Set 'StartDate' config value with a date object."""
        start_datestr = dateobj.strftime('%Y/%m/%d')
        self.config['DEFAULT']['StartDate'] = start_datestr
        self.save_config()

    def set_start_datestr(self, datestr):
        """Set 'StartDate' value directly as 'YY/MM/DD' string."""
        self.config['DEFAULT']['StartDate'] = datestr
        self.save_config()

    def get_progress(self):
        """Get 'Progress' value from config as an int"""
        return int(self.config['DEFAULT']['Progress'])

    def set_progress(self, progress):
        """Set 'Progress' value in config."""
        self.config['DEFAULT']['Progress'] = str(progress)
        self.save_config()

    def get_progress_date(self):
        """Returns 'Progress' + 'StartDate' as a date object"""
        return self.get_start_date() + timedelta(days=int(self.get_progress()))

    def get_progress_datestr(self):
        """Returns 'Progress' + 'StartDate' as a date string"""
        return self.get_progress_date().strftime('%Y/%m/%d')

