import click
import webbrowser
from dailybible.lib.esvclient import ESVClient
from dailybible.lib.appconfig import AppConfig

TEMP_HTML_PATH = '/tmp/dailybible_passages.html'

def write_to_temp_file(text):
    """Writes given string to temporary file."""
    with click.open_file(TEMP_HTML_PATH, 'w+') as file:
        file.write(text)

class DailyBibleReader(object):
    def __init__(self):
        self.esv_client = ESVClient()
        self.config = AppConfig()

    def set_output_format(self, output_format):
        """Sets the output format for the ESV client."""
        self.esv_client.output_format = output_format

    def read_passage(self):
        """Displays current Bible passages, in the console in or in a
        web browser.
        """
        start = self.config.get_start_datestr()
        current = self.config.get_progress_datestr()
        passages = self.esv_client.get(start, current)

        # display passages in the console or in the web browser
        if self.esv_client.output_format == 'plain-text':
            click.echo_via_pager(passages)
        else:
            # write passages to a temporary HTML file and open it in a
            # new web browser instance
            write_to_temp_file(passages)
            browser = None
            try:
                browser = webbrowser.get('firefox')
            except:
                browser = webbrowser.get()
            browser.open('file://%s' % (TEMP_HTML_PATH))

    def read_next_passage(self):
        """Increment progress and read next set of passages."""
        progress = int(self.config.get_progress()) + 1
        if (progress >= 365):
            progress = 0
        self.config.set_progress(progress)
        self.read_passage()

    def read_previous_passage(self):
        """Decrement progress and read previous set of passages."""
        progress = int(self.config.get_progress()) - 1
        if (progress < 0):
            progress = 0
        self.config.set_progress(progress)
        self.read_passage()

    def get_status_messages(self):
        """Returns a list of strings pertaining to reading plan
        progress.
        """
        start_date = self.config.get_start_date()
        friendly_start_date = start_date.strftime('%B %e, %Y')
        progress = self.config.get_progress()
        remaining = 365 - progress
        percent = int(progress/365.0 * 100.0)

        messages = [
            'Started reading plan on %s.' % (friendly_start_date),
            'You have read %d days worth of Bible passages.' % (progress),
            'You have %d days worth of passages left to read.' % (remaining),
            '%d%% complete.' % (percent)
        ]
        return messages

