#!/usr/bin/env python
# -*- coding: utf-8 -*-

# This work was created by participants in the DataONE project, and is
# jointly copyrighted by participating institutions in DataONE. For
# more information on DataONE, see our web site at http://dataone.org.
#
#   Copyright 2009-2012 DataONE
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#   http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

'''
:mod:`gmn_integration_tests`
============================

:Synopsis:
  Integration testing of ITK and GMN.

:Warning:
  This test deletes any existing objects and event log records on the
  destination GMN instance. For the tests to be able to run, ALLOW_UNIT_TESTS
  must be set to True in the settings_site.py file of the GMN instance being
  tested.

:Details:
  This test works by first putting the target GMN into a known state by deleting
  any existing objects and all event logs from the instance and then creating a
  set of test objects of which all object properties and exact contents are
  known. For each object, a set of fictitious events are stored in the event
  log. The test then runs through a series of tests where the GMN is queried,
  through the ITK, about all aspects of the object collection and the associated
  events and the results are compared with the known correct responses.

  GMN can handle storage of the object bytes itself ("managed" mode), or it can
  defer storage of the object bytes to another web server ("wrapped" mode). The
  mode is selectable on a per object basis. This test tests managed mode by
  default and can be set to test wrapped by specifying the --wrapped flag on
  the command line. For the wrapped mode tests to work, the test objects must be
  available on a web server. The location can be specified as a program
  argument.

:Created: 2010-06-14
:Author: DataONE (Dahl)
'''

# Stdlib.
import codecs
import datetime
#import dateutil
import glob
import hashlib
#import httplib
#import json
import logging
import optparse
import os
#import pprint
#import random
import re
#import stat
import StringIO
import sys
#import time
import unittest2
#import urllib
#import urlparse
#import uuid
#import xml.parsers.expat
#from xml.sax.saxutils import escape

# D1.
#import d1_client
import d1_client.mnclient
#import d1_client.systemmetadata
import d1_common.const
import d1_common.types.exceptions
import d1_common.types.generated.dataoneTypes as dataoneTypes
#import d1_common.util
import d1_common.date_time
import d1_common.url
#import d1_common.xml_compare

# App.
import gmn_test_client

# Constants.

# Test objects.
OBJECTS_TOTAL_DATA = 100
OBJECTS_UNIQUE_DATES = 99
OBJECTS_UNIQUE_DATE_AND_FORMAT_EML = 99
OBJECTS_PID_STARTSWITH_F = 3
OBJECTS_UNIQUE_DATE_AND_PID_STARTSWITH_F = 2
OBJECTS_CREATED_IN_90S = 32

# Event log.
#
# EVENTS_TOTAL is the number of records in the test_log.csv file. Because the
# tests themselves cause events to be generated, EVENTS_TOTAL is only correct
# just after the events have been injected. For the same reason, the other event
# counts include events that have been generated by the tests up to that point.
EVENTS_TOTAL = 452
EVENTS_TOTAL_1500 = EVENTS_TOTAL + 103
EVENTS_TOTAL_EVENT_UNI_TIME_IN_1990S = 117
EVENTS_DELETES_UNI_TIME_IN_1990S = 20

# Access control.
AUTH_PUBLIC_OBJECTS = 12
AUTH_SPECIFIC_USER = 'singing.3369'
AUTH_SPECIFIC_USER_OWNS = 19
AUTH_SPECIFIC_AND_OBJ_FORMAT = 19


def log_setup():
  # Set up logging.
  # We output everything to both file and stdout.
  logging.getLogger('').setLevel(logging.DEBUG)
  formatter = logging.Formatter('%(asctime)s %(levelname)-8s %(message)s',
                                '%y/%m/%d %H:%M:%S')
  file_logger = logging.FileHandler(os.path.splitext(__file__)[0] + '.log', 'a')
  file_logger.setFormatter(formatter)
  logging.getLogger('').addHandler(file_logger)
  console_logger = logging.StreamHandler(sys.stdout)
  console_logger.setFormatter(formatter)
  logging.getLogger('').addHandler(console_logger)


class GMNException(Exception):
  pass


class TestSequenceFunctions(unittest2.TestCase):
  def __init__(self, methodName='runTest'):
    unittest2.TestCase.__init__(self, methodName)


  def setUp(self):
    pass


  def assert_object_list_slice(self, object_list, start, count, total):
    self.assertEqual(object_list.start, start)
    self.assertEqual(object_list.count, count)
    self.assertEqual(object_list.total, total)
    # Check that the actual number of objects matches the count
    # provided in the slice.
    self.assertEqual(len(object_list.objectInfo), count)


  def assert_log_slice(self, log, start, count, total):
    self.assertEqual(log.start, start)
    self.assertEqual(log.count, count)
    self.assertEqual(log.total, total)
    # Check that the actual number of log records matches the count
    # provided in the slice.
    self.assertEqual(len(log.logEntry), count)


  def assert_response_headers(self, response):
    '''Required response headers are present.
    '''

    self.assertIn('Last-Modified', response)
    self.assertIn('Content-Length', response)
    self.assertIn('Content-Type', response)


  def assert_valid_date(self, date_str):
    self.assertTrue(datetime.datetime(*map(int, date_str.split('-'))))


  def find_valid_pid(self, client):
    '''Find the PID of an object that exists on the server.
    '''
    # Verify that there's at least one object on server.
    object_list = client.listObjects(
      vendorSpecific=self.include_subjects(gmn_test_client.GMN_TEST_SUBJECT_TRUSTED))
    self.assertTrue(object_list.count > 0, 'No objects to perform test on')
    # Get the first PID listed. The list is in random order.
    return object_list.objectInfo[0].identifier.value()


  def generate_sysmeta(self, pid, size, md5, now, owner):
    sysmeta = dataoneTypes.systemMetadata()
    sysmeta.serialVersion = 1
    sysmeta.identifier = pid
    sysmeta.formatId = 'eml://ecoinformatics.org/eml-2.0.0'
    sysmeta.size = size
    sysmeta.submitter = owner
    sysmeta.rightsHolder = owner
    sysmeta.checksum = dataoneTypes.checksum(md5)
    sysmeta.checksum.algorithm = 'MD5'
    sysmeta.dateUploaded = now
    sysmeta.dateSysMetadataModified = now
    sysmeta.originMemberNode = 'MN1'
    sysmeta.authoritativeMemberNode = 'MN1'
    return sysmeta


  def generate_access_policy(self, access_rules):
    accessPolicy = dataoneTypes.accessPolicy()
    for access_rule in access_rules:
      accessRule = dataoneTypes.AccessRule()
      for subject in access_rule[0]:
        accessRule.subject.append(subject)
      for permission in access_rule[1]:
        permission_pyxb = dataoneTypes.Permission(permission)
        accessRule.permission.append(permission_pyxb)
      accessPolicy.append(accessRule)
    return accessPolicy


  def generate_test_object(self, pid):
    '''Generate a random, small, SciObj / SysMeta pair'''
    # Create a small test object containing only the pid.
    sciobj = pid.encode('utf-8')
    # Create corresponding System Metadata for the test object.
    size = len(sciobj)
    # hashlib.md5 can't hash a Unicode string. If it did, we would get a hash
    # of the internal Python encoding for the string. So we maintain sciobj
    # as a UTF-8 string.
    md5 = hashlib.md5(sciobj).hexdigest()
    now = datetime.datetime.now()
    sysmeta = self.generate_sysmeta(pid, size, md5, now,
                                    gmn_test_client.GMN_TEST_SUBJECT_PUBLIC)
    return sciobj, sysmeta


  def include_subjects(self, subjects):
    if isinstance(subjects, basestring):
      subjects = [subjects]
    return {'VENDOR_INCLUDE_SUBJECTS': '\t'.join(subjects)}


  def has_public_object_list(self):
    client = gmn_test_client.GMNTestClient(self.options.gmn_url)
    return eval(client.get_setting('PUBLIC_OBJECT_LIST'))

  # ============================================================================
  # Preparation.
  # ============================================================================

  def test_1000_A(self):
    '''GMN must be in debug mode when running unit tests.'''
    client = gmn_test_client.GMNTestClient(self.options.gmn_url)
    self.assertEqual(client.get_setting('GMN_DEBUG'), 'True')


  def test_1000_B(self):
    '''GMN must be set to allow running destructive integration tests.'''
    client = gmn_test_client.GMNTestClient(self.options.gmn_url)
    self.assertEqual(client.get_setting('ALLOW_INTEGRATION_TESTS'), 'True')


  def test_1020_A(self):
    '''Delete all objects.'''
    client = gmn_test_client.GMNTestClient(self.options.gmn_url)
    client.delete_all_objects(
      headers=self.include_subjects(gmn_test_client.GMN_TEST_SUBJECT_TRUSTED))


  def test_1020_B(self):
    '''Object collection is empty.'''
    client = d1_client.mnclient.MemberNodeClient(self.options.gmn_url)
    object_list = client.listObjects(
      vendorSpecific=self.include_subjects(gmn_test_client.GMN_TEST_SUBJECT_TRUSTED))
    self.assert_object_list_slice(object_list, 0, 0, 0)


  def test_1020_C(self):
    '''Delete all replication requests.'''
    client = gmn_test_client.GMNTestClient(self.options.gmn_url)
    client.clear_replication_queue(
      headers=self.include_subjects(gmn_test_client.GMN_TEST_SUBJECT_TRUSTED))

  # ----------------------------------------------------------------------------
  # Set up test objects. Also checks create()
  # ----------------------------------------------------------------------------

  def test_1050_A(self):
    '''Populate MN with set of test objects.
    Uses the internal diagnostics create() and does not test create permissions.
    '''
    client = gmn_test_client.GMNTestClient(self.options.gmn_url)
    for sysmeta_path in sorted(glob.glob(os.path.join(self.options.obj_path,
                                                      '*.sysmeta'))):
      # Get name of corresponding object and open it.
      object_path = re.match(r'(.*)\.sysmeta', sysmeta_path).group(1)
      object_file = open(object_path, 'r')

      # The pid is stored in the sysmeta.
      sysmeta_file = open(sysmeta_path, 'r')
      sysmeta_xml = sysmeta_file.read()
      sysmeta_obj = dataoneTypes.CreateFromDocument(sysmeta_xml)
      sysmeta_obj.rightsHolder = 'test_user_1'

      headers = self.include_subjects('test_user_1')
      headers.update({'VENDOR_TEST_OBJECT': 1})

      if self.options.wrapped:
        vendor_specific = {
          'VENDOR_GMN_REMOTE_URL': self.options.obj_url + '/' + \
          d1_common.url.encodePathElement(
            d1_common.url.encodePathElement(sysmeta_obj.identifier.value()))
        }
        headers.update(vendor_specific)

      client.create(sysmeta_obj.identifier.value(),
                    object_file, sysmeta_obj,
                    vendorSpecific=headers)


  def test_1050_B(self):
    '''Object collection is populated.
    '''
    client = d1_client.mnclient.MemberNodeClient(self.options.gmn_url)
    # Get object collection.
    object_list = client.listObjects(count=d1_common.const.MAX_LISTOBJECTS,
      vendorSpecific=self.include_subjects(gmn_test_client.GMN_TEST_SUBJECT_TRUSTED))
    # Check header.
    self.assert_object_list_slice(object_list, 0, OBJECTS_TOTAL_DATA, OBJECTS_TOTAL_DATA)


  # ----------------------------------------------------------------------------
  # Set up test event log.
  # ----------------------------------------------------------------------------

  def test_1100_A(self):
    '''Clear event log.
    '''
    client = gmn_test_client.GMNTestClient(self.options.gmn_url)
    client.delete_event_log(
      headers=self.include_subjects(gmn_test_client.GMN_TEST_SUBJECT_TRUSTED))


  def test_1100_B(self):
    '''Event log is empty.
    '''
    '''Object collection is empty.
    '''
    client = d1_client.mnclient.MemberNodeClient(self.options.gmn_url)
    logRecords = client.getLogRecords()
    self.assertEqual(len(logRecords.logEntry), 0)


  def test_1100_C(self):
    '''Inject a set of fictitious events for each object.
    '''
    csv_file = open('test_log.csv', 'rb')
    client = gmn_test_client.GMNTestClient(self.options.gmn_url)
    client.inject_fictional_event_log(csv_file,
      headers=self.include_subjects(gmn_test_client.GMN_TEST_SUBJECT_TRUSTED))

  def test_1100_D(self):
    '''Event log is populated.
    '''
    client = d1_client.mnclient.MemberNodeClient(self.options.gmn_url)
    logRecords = client.getLogRecords(count=d1_common.const.MAX_LISTOBJECTS,
      vendorSpecific=self.include_subjects(
        gmn_test_client.GMN_TEST_SUBJECT_TRUSTED))
    self.assertEqual(len(logRecords.logEntry), EVENTS_TOTAL)
    found = False
    for o in logRecords.logEntry:
      if o.identifier.value() == 'hdl:10255/dryad.654/mets.xml' \
                                 and o.event == 'create':
        found = True
        break
    self.assertTrue(found)

  # ============================================================================
  # Read API
  # ============================================================================

  # ----------------------------------------------------------------------------
  # get()
  # ----------------------------------------------------------------------------

  def test_1200(self):
    '''get(): Successful retrieval of valid object.
    '''
    client = d1_client.mnclient.MemberNodeClient(self.options.gmn_url)
    response = client.get('10Dappend2.txt',
      vendorSpecific=self.include_subjects(gmn_test_client.GMN_TEST_SUBJECT_TRUSTED))
    headers = response.getheaders()
    self.assertTrue(('content-length', '1982') in headers)
    self.assertTrue(('dataone-checksum', 'MD5,ed387674851ba80bd2d3c6c42f335cf7') in headers)
    self.assertTrue(('dataone-formatid', 'eml://ecoinformatics.org/eml-2.0.0') in headers)
    self.assertTrue(('last-modified', '1977-03-09T00:12:05') in headers)
    self.assertTrue(('content-type', 'text/xml') in headers)


  def test_1210(self):
    '''get(): 404 NotFound when attempting to get non-existing object.
    '''
    client = d1_client.mnclient.MemberNodeClient(self.options.gmn_url)
    self.assertRaises(d1_common.types.exceptions.NotFound, client.get,
                      '_invalid_pid_')


  def test_1220(self):
    '''get(): Read from MN and do byte-by-byte comparison with local copies.
    '''
    client = d1_client.mnclient.MemberNodeClient(self.options.gmn_url, timeout=60)

    for sysmeta_path in sorted(glob.glob(os.path.join(self.options.obj_path,
                                                      '*.sysmeta'))):
      object_path = re.match(r'(.*)\.sysmeta', sysmeta_path).group(1)
      pid = d1_common.url.decodePathElement(os.path.basename(object_path))
      #sysmeta_xml_disk = open(sysmeta_path, 'r').read()
      object_str_disk = open(object_path, 'rb').read()
      #sysmeta_xml_d1 = client.getSystemMetadata(pid).read()
      object_str_d1 = client.get(pid,
        vendorSpecific=self.include_subjects(gmn_test_client.GMN_TEST_SUBJECT_TRUSTED)) \
        .read(1024 ** 2)
      self.assertEqual(object_str_disk, object_str_d1)

  # ----------------------------------------------------------------------------
  # getSystemMetadata()
  # ----------------------------------------------------------------------------

  def test_1250(self):
    '''getSystemMetadata(): Successful retrieval of SysMeta of valid object.
    '''
    client = d1_client.mnclient.MemberNodeClient(self.options.gmn_url)
    response = client.getSystemMetadata('10Dappend2.txt',
      vendorSpecific=self.include_subjects(gmn_test_client.GMN_TEST_SUBJECT_TRUSTED))
    self.assertTrue(response)


  def test_1260(self):
    '''getSystemMetadata(): 404 NotFound when attempting to get non-existing SysMeta.
    '''
    client = d1_client.mnclient.MemberNodeClient(self.options.gmn_url)
    self.assertRaises(d1_common.types.exceptions.NotFound,
      client.getSystemMetadata,
      '_invalid_pid_',
      vendorSpecific=self.include_subjects(gmn_test_client.GMN_TEST_SUBJECT_TRUSTED))

  # ----------------------------------------------------------------------------
  # describe()
  # ----------------------------------------------------------------------------

  def test_1290(self):
    '''MNStorage.describe(): Returns valid header for valid object.
    '''
    client = d1_client.mnclient.MemberNodeClient(self.options.gmn_url)
    # Find the PID for a random object that exists on the server.
    pid = self.find_valid_pid(client)
    # Get header information for object.
    info = client.describe(pid,
      vendorSpecific=self.include_subjects(gmn_test_client.GMN_TEST_SUBJECT_TRUSTED))
    self.assertTrue(re.search(r'dataone-formatid', str(info)))
    self.assertTrue(re.search(r'content-length', str(info)))
    self.assertTrue(re.search(r'last-modified', str(info)))
    self.assertTrue(re.search(r'dataone-checksum', str(info)))

  # ----------------------------------------------------------------------------
  # listObjects()
  # ----------------------------------------------------------------------------

  def test_1300(self):
    '''listObjects(): Read complete object collection and compare with values stored in local SysMeta files.
    '''
    # Get object collection.
    client = d1_client.mnclient.MemberNodeClient(self.options.gmn_url, timeout=60)
    object_list = client.listObjects(count=d1_common.const.MAX_LISTOBJECTS,
      vendorSpecific=self.include_subjects(gmn_test_client.GMN_TEST_SUBJECT_TRUSTED))

    # Loop through our local test objects.
    for sysmeta_path in sorted(glob.glob(os.path.join(self.options.obj_path,
                                                      '*.sysmeta'))):
      # Get name of corresponding object and check that it exists on disk.
      object_path = re.match(r'(.*)\.sysmeta', sysmeta_path).group(1)
      self.assertTrue(os.path.exists(object_path))
      # Get pid for object.
      pid = d1_common.url.decodePathElement(os.path.basename(object_path))
      # Get sysmeta xml for corresponding object from disk.
      sysmeta_file = open(sysmeta_path, 'rb')
      sysmeta_xml = sysmeta_file.read()
      sysmeta_obj = dataoneTypes.CreateFromDocument(sysmeta_xml)

      # Get corresponding object from objectList.
      found = False
      for object_info in object_list.objectInfo:
        if object_info.identifier.value() == sysmeta_obj.identifier.value():
          found = True
          break;

      self.assertTrue(found,
        'Couldn\'t find object with pid "{0}"'.format(sysmeta_obj.identifier))

      self.assertEqual(object_info.identifier.value(),
                       sysmeta_obj.identifier.value(), sysmeta_path)
      self.assertEqual(object_info.formatId,
                       sysmeta_obj.formatId, sysmeta_path)
      self.assertEqual(object_info.dateSysMetadataModified,
                       sysmeta_obj.dateSysMetadataModified, sysmeta_path)
      self.assertEqual(object_info.size, sysmeta_obj.size, sysmeta_path)
      self.assertEqual(object_info.checksum.value(),
                       sysmeta_obj.checksum.value(), sysmeta_path)
      self.assertEqual(object_info.checksum.algorithm,
                       sysmeta_obj.checksum.algorithm, sysmeta_path)


  def test_1310(self):
    '''listObjects(): Get object count.
    '''
    client = d1_client.mnclient.MemberNodeClient(self.options.gmn_url)

    object_list = client.listObjects(start=0, count=0,
      vendorSpecific=self.include_subjects(gmn_test_client.GMN_TEST_SUBJECT_TRUSTED))
    self.assert_object_list_slice(object_list, 0, 0, OBJECTS_TOTAL_DATA)


  def test_1320(self):
    '''listObjects(): Slicing: Starting at 0 and getting half of the available objects.
    '''
    object_cnt_half = OBJECTS_TOTAL_DATA / 2
    # Starting at 0 and getting half of the available objects.
    client = d1_client.mnclient.MemberNodeClient(self.options.gmn_url)
    object_list = client.listObjects(start=0, count=object_cnt_half,
      vendorSpecific=self.include_subjects(gmn_test_client.GMN_TEST_SUBJECT_TRUSTED))
    self.assert_object_list_slice(object_list, 0, object_cnt_half,
                       OBJECTS_TOTAL_DATA)


  def test_1330(self):
    '''listObjects(): Slicing: Starting at object_cnt_half and requesting more objects
    than there are.
    '''
    object_cnt_half = OBJECTS_TOTAL_DATA / 2
    client = d1_client.mnclient.MemberNodeClient(self.options.gmn_url)
    object_list = client.listObjects(start=object_cnt_half,
      count=d1_common.const.MAX_LISTOBJECTS,
      vendorSpecific=self.include_subjects(gmn_test_client.GMN_TEST_SUBJECT_TRUSTED))
    self.assert_object_list_slice(object_list, object_cnt_half, object_cnt_half,
                       OBJECTS_TOTAL_DATA)


  def test_1340(self):
    '''listObjects(): Slicing: Starting above number of objects that we have.
    '''
    client = d1_client.mnclient.MemberNodeClient(self.options.gmn_url)
    object_list = client.listObjects(start=OBJECTS_TOTAL_DATA * 2, count=1,
      vendorSpecific=self.include_subjects(gmn_test_client.GMN_TEST_SUBJECT_TRUSTED))
    self.assert_object_list_slice(object_list, OBJECTS_TOTAL_DATA * 2, 0,
                       OBJECTS_TOTAL_DATA)


  def test_1360(self):
    '''listObjects(): Date range query: Get all objects from the 1990s.
    '''
    client = d1_client.mnclient.MemberNodeClient(self.options.gmn_url)

    object_list = client.listObjects(count=d1_common.const.MAX_LISTOBJECTS,
      fromDate=datetime.datetime(1990, 1, 1),
      toDate=datetime.datetime(1999, 12, 31),
      vendorSpecific=self.include_subjects(gmn_test_client.GMN_TEST_SUBJECT_TRUSTED))
    self.assert_object_list_slice(object_list, 0,
                       OBJECTS_CREATED_IN_90S, OBJECTS_CREATED_IN_90S)


  def test_1370(self):
    '''listObjects(): Date range query: Get first 10 objects from the 1990s.
    '''
    client = d1_client.mnclient.MemberNodeClient(self.options.gmn_url)

    object_list = client.listObjects(start=0, count=10,
      fromDate=datetime.datetime(1990, 1, 1),
      toDate=datetime.datetime(1999, 12, 31),
      vendorSpecific=self.include_subjects(gmn_test_client.GMN_TEST_SUBJECT_TRUSTED))
    self.assert_object_list_slice(object_list, 0, 10, OBJECTS_CREATED_IN_90S)


  def test_1380(self):
    '''listObjects(): Date range query: Get 10 first objects from the 1990s, filtered by objectFormat.
    '''
    client = d1_client.mnclient.MemberNodeClient(self.options.gmn_url)

    object_list = client.listObjects(start=0, count=10,
      fromDate=datetime.datetime(1990, 1, 1),
      toDate=datetime.datetime(1999, 12, 31),
      objectFormat='eml://ecoinformatics.org/eml-2.0.0',
      vendorSpecific=self.include_subjects(gmn_test_client.GMN_TEST_SUBJECT_TRUSTED))
    self.assert_object_list_slice(object_list, 0, 10, OBJECTS_CREATED_IN_90S)


  def test_1390(self):
    '''listObjects(): Date range query: Get 10 first objects from non-existing date range.
    '''
    client = d1_client.mnclient.MemberNodeClient(self.options.gmn_url)

    object_list = client.listObjects(start=0, count=10,
      fromDate=datetime.datetime(2500, 1, 1),
      toDate=datetime.datetime(2500, 12, 31),
      objectFormat='eml://ecoinformatics.org/eml-2.0.0',
      vendorSpecific=self.include_subjects(gmn_test_client.GMN_TEST_SUBJECT_TRUSTED))
    self.assert_object_list_slice(object_list, 0, 0, 0)


  def test_1400(self):
    '''listObjects(): Returns all objects when called by trusted user.
    '''
    client = d1_client.mnclient.MemberNodeClient(self.options.gmn_url)
    object_list = client.listObjects(count=d1_common.const.MAX_LISTOBJECTS,
      vendorSpecific=self.include_subjects(gmn_test_client.GMN_TEST_SUBJECT_TRUSTED))
    self.assertEqual(object_list.count, OBJECTS_TOTAL_DATA)


  def test_1410(self):
    '''listObjects(): Returns only public objects when called by public user.
    '''
    # This test can only run if public access has been enabled for listObjects.
    if not self.has_public_object_list():
      return
    client = d1_client.mnclient.MemberNodeClient(self.options.gmn_url)
    object_list = client.listObjects(count=d1_common.const.MAX_LISTOBJECTS,
      vendorSpecific=self.include_subjects(gmn_test_client.GMN_TEST_SUBJECT_PUBLIC))
    self.assertEqual(object_list.count, AUTH_PUBLIC_OBJECTS)


  def test_1420(self):
    '''listObjects(): Returns only public objects when called by unknown user.
    '''
    # This test can only run if public access has been enabled for listObjects.
    if not self.has_public_object_list():
      return
    client = d1_client.mnclient.MemberNodeClient(self.options.gmn_url)
    object_list = client.listObjects(count=d1_common.const.MAX_LISTOBJECTS,
      vendorSpecific=self.include_subjects('unknown user'))
    self.assertEqual(object_list.count, AUTH_PUBLIC_OBJECTS)


  def test_1430(self):
    '''listObjects(): returns only public + specific user's objects
    '''
    # This test can only run if public access has been enabled for listObjects.
    if not self.has_public_object_list():
      return
    client = d1_client.mnclient.MemberNodeClient(self.options.gmn_url)
    object_list = client.listObjects(count=d1_common.const.MAX_LISTOBJECTS,
      vendorSpecific=self.include_subjects(AUTH_SPECIFIC_USER))
    self.assertEqual(object_list.count, AUTH_SPECIFIC_USER_OWNS)


  def test_1440(self):
    '''listObjects(): slicing + specific user
    '''
    # This test can only run if public access has been enabled for listObjects.
    if not self.has_public_object_list():
      return
    client = d1_client.mnclient.MemberNodeClient(self.options.gmn_url)
    object_list = client.listObjects(count=5,
      vendorSpecific=self.include_subjects(AUTH_SPECIFIC_USER))
    self.assert_object_list_slice(object_list, 0, 5, AUTH_SPECIFIC_USER_OWNS)


  def test_1450(self):
    '''listObjects(): slicing + specific user + objectFormat
    '''
    # This test can only run if public access has been enabled for listObjects.
    if not self.has_public_object_list():
      return
    client = d1_client.mnclient.MemberNodeClient(self.options.gmn_url)
    object_list = client.listObjects(count=5,
      objectFormat='eml://ecoinformatics.org/eml-2.0.0',
      vendorSpecific=self.include_subjects(AUTH_SPECIFIC_USER))
    self.assert_object_list_slice(object_list, 0, 5, AUTH_SPECIFIC_AND_OBJ_FORMAT)

  # ----------------------------------------------------------------------------
  # getLogRecords()
  # ----------------------------------------------------------------------------

  def test_1500(self):
    '''getLogRecords(): Get event count
    '''
    client = d1_client.mnclient.MemberNodeClient(self.options.gmn_url)
    log = client.getLogRecords(start=0, count=0,
       vendorSpecific=self.include_subjects(
       gmn_test_client.GMN_TEST_SUBJECT_TRUSTED))
    self.assert_log_slice(log, 0, 0, EVENTS_TOTAL_1500)


  def test_1510(self):
    '''getLogRecords(): Slicing: Starting at 0 and getting half of the available events.
    '''
    object_cnt_half = EVENTS_TOTAL / 2
    # Starting at 0 and getting half of the available objects.
    client = d1_client.mnclient.MemberNodeClient(self.options.gmn_url)
    log = client.getLogRecords(start=0, count=object_cnt_half,
      vendorSpecific=self.include_subjects(
      gmn_test_client.GMN_TEST_SUBJECT_TRUSTED))
    self.assert_log_slice(log, 0, object_cnt_half,
                          EVENTS_TOTAL_1500)


  def test_1520(self):
    '''getLogRecords(): Slicing: From center and more than are available
    '''
    object_cnt_half = EVENTS_TOTAL_1500 / 2
    client = d1_client.mnclient.MemberNodeClient(self.options.gmn_url)
    log = client.getLogRecords(start=object_cnt_half,
      count=d1_common.const.MAX_LISTOBJECTS,
      vendorSpecific=self.include_subjects(
        gmn_test_client.GMN_TEST_SUBJECT_TRUSTED))
    self.assert_log_slice(log, object_cnt_half,
                          EVENTS_TOTAL_1500 - object_cnt_half,
                          EVENTS_TOTAL_1500)


  def test_1530(self):
    '''getLogRecords(): Slicing: Starting above number of events that are available.
    '''
    client = d1_client.mnclient.MemberNodeClient(self.options.gmn_url)
    log = client.getLogRecords(start=EVENTS_TOTAL_1500 * 2, count=1,
      vendorSpecific=self.include_subjects(
        gmn_test_client.GMN_TEST_SUBJECT_TRUSTED))
    self.assert_log_slice(log, EVENTS_TOTAL_1500 * 2, 0,
                       EVENTS_TOTAL_1500)


  def test_1550(self):
    '''getLogRecords(): Date range query: Get all events from the 1990s.
    '''
    client = d1_client.mnclient.MemberNodeClient(self.options.gmn_url)

    log = client.getLogRecords(count=d1_common.const.MAX_LISTOBJECTS,
      fromDate=datetime.datetime(1990, 1, 1),
      toDate=datetime.datetime(1999, 12, 31),
      vendorSpecific=self.include_subjects(gmn_test_client.GMN_TEST_SUBJECT_TRUSTED))
    self.assert_log_slice(log, 0, EVENTS_TOTAL_EVENT_UNI_TIME_IN_1990S,
                          EVENTS_TOTAL_EVENT_UNI_TIME_IN_1990S)


  def test_1560(self):
    '''getLogRecords(): Date range query: Get first 10 objects from the 1990s.
    '''
    client = d1_client.mnclient.MemberNodeClient(self.options.gmn_url)
    log = client.getLogRecords(start=0, count=10,
      fromDate=datetime.datetime(1990, 1, 1),
      toDate=datetime.datetime(1999, 12, 31),
      vendorSpecific=self.include_subjects(gmn_test_client.GMN_TEST_SUBJECT_TRUSTED))
    self.assert_log_slice(log, 0, 10, EVENTS_TOTAL_EVENT_UNI_TIME_IN_1990S)


  def test_1570(self):
    '''getLogRecords(): Date range query: Get all events from the 1990s, filtered by event type.
    '''
    client = d1_client.mnclient.MemberNodeClient(self.options.gmn_url)
    log = client.getLogRecords(start=0, count=d1_common.const.MAX_LISTOBJECTS,
      fromDate=datetime.datetime(1990, 1, 1),
      toDate=datetime.datetime(1999, 12, 31),
      event='delete',
      vendorSpecific=self.include_subjects(gmn_test_client.GMN_TEST_SUBJECT_TRUSTED))
    self.assert_log_slice(log, 0, EVENTS_DELETES_UNI_TIME_IN_1990S,
                          EVENTS_DELETES_UNI_TIME_IN_1990S)


  def test_1580(self):
    '''getLogRecords(): Date range query: Get 10 first events from non-existing date range.
    '''
    client = d1_client.mnclient.MemberNodeClient(self.options.gmn_url)

    log = client.getLogRecords(start=0, count=d1_common.const.MAX_LISTOBJECTS,
      fromDate=datetime.datetime(2500, 1, 1),
      toDate=datetime.datetime(2500, 12, 31),
      vendorSpecific=self.include_subjects(gmn_test_client.GMN_TEST_SUBJECT_TRUSTED))
    self.assert_log_slice(log, 0, 0, 0)

  # ----------------------------------------------------------------------------
  # getChecksum()
  # ----------------------------------------------------------------------------

  def _get_checksum_test(self, pid, checksum, algorithm):
    client = d1_client.mnclient.MemberNodeClient(self.options.gmn_url)
    checksum_obj = client.getChecksum(pid, checksumAlgorithm=algorithm,
      vendorSpecific=self.include_subjects(
        gmn_test_client.GMN_TEST_SUBJECT_TRUSTED))
    self.assertTrue(isinstance(checksum_obj, dataoneTypes.Checksum))
    self.assertEqual(checksum, checksum_obj.value())
    self.assertEqual(algorithm, checksum_obj.algorithm)


  def test_1600(self):
    '''getChecksum(): MD5'''
    pid = 'Drugeffect.xls'
    checksum = '916a377112e3d4ed5812f8493a271966'
    algorithm = 'MD5'
    self._get_checksum_test(pid, checksum, algorithm)


  def test_1610(self):
    '''getChecksum(): SHA-1'''
    pid = 'emerson.app'
    checksum = '20b95b4c68c949f1a373efd3a4d612557d8e49b1'
    algorithm = 'SHA-1'
    self._get_checksum_test(pid, checksum, algorithm)


  def test_1620(self):
    '''getChecksum(): Unsupported algorithm returns InvalidRequest exception'''
    pid = 'FigS2_Hsieh.pdf'
    algorithm = 'INVALID_ALGORITHM'
    self.assertRaises(d1_common.types.exceptions.InvalidRequest,
                      self._get_checksum_test, pid, '', algorithm)


  def test_1630(self):
    '''getChecksum(): Non-existing object raises NotFound exception'''
    pid = 'non-existing-pid'
    algorithm = 'MD5'
    self.assertRaises(d1_common.types.exceptions.NotFound,
                      self._get_checksum_test, pid, '', algorithm)

  # ----------------------------------------------------------------------------
  # systemMetadataChanged()
  # ----------------------------------------------------------------------------

  def test_1700(self):
    '''systemMetadataChanged(): fails when called with invalid PID'''
    client = d1_client.mnclient.MemberNodeClient(self.options.gmn_url)
    self.assertRaises(d1_common.types.exceptions.NotFound,
      client.systemMetadataChanged, '_bogus_pid_', 1,
        d1_common.date_time.utc_now(),
        vendorSpecific=self.include_subjects(gmn_test_client.GMN_TEST_SUBJECT_TRUSTED))


  def test_1710(self):
    '''systemMetadataChanged(): succeeds when called with valid PID'''
    client = d1_client.mnclient.MemberNodeClient(self.options.gmn_url)
    client.systemMetadataChanged('fitch2.mc', 1, d1_common.date_time.utc_now(),
      vendorSpecific=self.include_subjects(gmn_test_client.GMN_TEST_SUBJECT_TRUSTED))


  def test_1720(self):
    '''systemMetadataChanged(): denies access to subjects other that CNs'''
    client = d1_client.mnclient.MemberNodeClient(self.options.gmn_url)
    self.assertRaises(d1_common.types.exceptions.NotAuthorized,
      client.systemMetadataChanged, 'fitch2.mc', 1,
        d1_common.date_time.utc_now(),
        vendorSpecific=self.include_subjects(gmn_test_client.GMN_TEST_SUBJECT_PUBLIC))

  # ----------------------------------------------------------------------------
  # synchronizationFailed()
  # ----------------------------------------------------------------------------

  def test_1800(self):
    '''MNRead.synchronizationFailed() with valid error returns 200 OK.
    '''
    # This test does not test if GMN actually does anything with the message
    # passed to the synchronizationFailed() method. There is currently no way
    # for the test to reach that information.
    pid = '12Cpaup.txt'
    msg = 'TEST MESSAGE FROM GMN_INTEGRATION_TESTER'
    exception = d1_common.types.exceptions.SynchronizationFailed(0, msg, pid)
    client = d1_client.mnclient.MemberNodeClient(self.options.gmn_url)
    client.synchronizationFailed(exception,
      vendorSpecific=self.include_subjects(gmn_test_client.GMN_TEST_SUBJECT_TRUSTED))


  def test_1810(self):
    '''MNRead.synchronizationFailed() from untrusted subject raises NotAuthorized.
    '''
    pid = '12Cpaup.txt'
    msg = 'TEST MESSAGE FROM GMN_INTEGRATION_TESTER'
    exception = d1_common.types.exceptions.SynchronizationFailed(0, msg, pid)
    client = d1_client.mnclient.MemberNodeClient(self.options.gmn_url)
    self.assertRaises(d1_common.types.exceptions.NotAuthorized,
                      client.synchronizationFailed, exception)


# Disabled because, in v1, InvalidRequest is not a valid response for
# MNRead.synchronizationFailed(). MNRead.synchronizationFailed() must return
# a 200 OK even if there is an issue with the call.
#   def test_1820(self):
#     '''MNRead.synchronizationFailed() with invalid XML document raises InvalidRequest.
#     '''
#     class InvalidException():
#       def serialize(self):
#         return 'INVALID SERIALIZED DATAONE EXCEPTION'
#
#     client = d1_client.mnclient.MemberNodeClient(self.options.gmn_url)
#     self.assertRaises(d1_common.types.exceptions.InvalidRequest,
#                       client.synchronizationFailed,
#                       InvalidException(),
#                       vendorSpecific=self.include_subjects(gmn_test_client.GMN_TEST_SUBJECT_TRUSTED))

  def test_1830(self):
    '''MNRead.synchronizationFailed() with invalid XML document returns 200 OK.
    '''
    class InvalidException():
      def serialize(self):
        return 'INVALID SERIALIZED DATAONE EXCEPTION'

    client = d1_client.mnclient.MemberNodeClient(self.options.gmn_url)
    print client.synchronizationFailed(InvalidException(),
      vendorSpecific=self.include_subjects(gmn_test_client.GMN_TEST_SUBJECT_TRUSTED))



# ============================================================================
  # Misc.
  # ============================================================================

  # ----------------------------------------------------------------------------
  # node
  # ----------------------------------------------------------------------------

  def test_1850(self):
    '''MNCore.getCapabilities(): Returns a valid Node Registry document.
    '''
    client = d1_client.mnclient.MemberNodeClient(self.options.gmn_url)
    node = client.getCapabilities()
    self.assertTrue(isinstance(node, dataoneTypes.Node))


  def _generate_identifier(self):
    client = d1_client.mnclient.MemberNodeClient(self.options.gmn_url)
    fragment = 'test_fragment'
    identifier = client.generateIdentifier('UUID', fragment)
    self.assertTrue(identifier.value().startswith(fragment))
    self.assertTrue(len(identifier.value()) > len(fragment))
    return identifier.value()


  def test_1860_A(self):
    '''MNStorage.generateIdentifier(): Returns a valid identifier that matches scheme and fragment'''
    self._generate_identifier()


  def test_1860_B(self):
    '''MNStorage.generateIdentifier(): Returns a different, valid identifier when called second time'''
    pid1 = self._generate_identifier()
    pid2 = self._generate_identifier()
    self.assertNotEqual(pid1, pid2)

  # ----------------------------------------------------------------------------
  # MNReplication.replicate()
  # ----------------------------------------------------------------------------

  def test_1900(self):
    '''MNReplication.replicate(): Request to replicate new object returns 200 OK.
    Does NOT check if GMN acts on the request and actually performs the replication.
    '''
    known_pid = 'new_pid'
    scidata, sysmeta = self.generate_test_object(known_pid)
    client = d1_client.mnclient.MemberNodeClient(self.options.gmn_url)
    client.replicate(sysmeta, 'test_source_node',
      vendorSpecific=self.include_subjects(gmn_test_client.GMN_TEST_SUBJECT_TRUSTED))


  def test_1910(self):
    '''MNReplication.replicate(): Request to replicate existing object raises IdentifierNotUnique.
    Does NOT check if GMN acts on the request and actually performs the replication.
    '''
    known_pid = 'AnserMatrix.htm'
    scidata, sysmeta = self.generate_test_object(known_pid)
    client = d1_client.mnclient.MemberNodeClient(self.options.gmn_url)
    self.assertRaises(d1_common.types.exceptions.IdentifierNotUnique,
      client.replicate, sysmeta, 'test_source_node',
      vendorSpecific=self.include_subjects(gmn_test_client.GMN_TEST_SUBJECT_TRUSTED))


  def test_1920(self):
    '''MNReplication.replicate(): Request from non-trusted subject returns NotAuthorized.
    Does NOT check if GMN acts on the request and actually performs the replication.
    '''
    known_pid = 'new_pid_2'
    scidata, sysmeta = self.generate_test_object(known_pid)
    client = d1_client.mnclient.MemberNodeClient(self.options.gmn_url)
    self.assertRaises(d1_common.types.exceptions.NotAuthorized,
      client.replicate, sysmeta, 'test_source_node')


  # ----------------------------------------------------------------------------
  # MNStorage.update()
  # ----------------------------------------------------------------------------

#  def test_2000(self):
#    '''Update System Metadata.
#    '''
#    pid = '12Cpaup.txt'
#
#    # Generate a new System Metadata object with Access Policy.
#    sysmeta = self.generate_sysmeta(pid, 123, 'baadf00d',
#                                    datetime.datetime(1976, 7, 8),
#                                    gmn_test_client.GMN_TEST_SUBJECT_TRUSTED)
#
#    access_policy_spec = (
#      (('test_user_1',), ('read',)),
#      (('test_user_2',), ('read',))
#    )
#
#    sysmeta.accessPolicy = self.generate_access_policy(access_policy_spec)
#
#    sysmeta.rightsHolder = 'test_user_1'
#
#    # Serialize System Metadata to XML.
#    sysmeta_xml = sysmeta.toxml()
#    mime_multipart_files = [
#      ('sysmeta','systemmetadata.abc', sysmeta_xml.encode('utf-8')),
#    ]
#
#    # POST to /meta/pid.
#    test_test_1060_update_sysmeta_url = urlparse.urljoin('/v1/meta/',
#      d1_common.url.encodePathElement(pid))
#
#    root = gmn_test_client.GMNTestClient(self.options.gmn_url)
#    response = root.POST(
#      test_test_1060_update_sysmeta_url, files=mime_multipart_files,
#      headers=self.include_subjects(gmn_test_client.GMN_TEST_SUBJECT_TRUSTED))
#    self.assertEqual(response.status, 200)

  def test_2010_A(self):
    '''MNStorage.update(): Creating a new object that obsoletes another object.
    '''
    new_pid = 'update_object_pid_1'
    old_pid = 'AnserMatrix.htm'
    sci_obj, sys_meta = self.generate_test_object(new_pid)
    client = d1_client.mnclient.MemberNodeClient(self.options.gmn_url)
    client.update(old_pid, StringIO.StringIO(sci_obj), new_pid, sys_meta,
      vendorSpecific=self.include_subjects('test_user_1'))


  def test_2010_B(self):
    '''MNStorage.update(): Attempt to update an obsoleted object raises InvalidRequest.
    '''
    new_pid = 'update_object_pid_2'
    old_pid = 'AnserMatrix.htm'
    sci_obj, sys_meta = self.generate_test_object(new_pid)
    client = d1_client.mnclient.MemberNodeClient(self.options.gmn_url)
    self.assertRaises(d1_common.types.exceptions.InvalidRequest,
      client.update, old_pid, StringIO.StringIO(sci_obj), new_pid, sys_meta,
      vendorSpecific=self.include_subjects('test_user_1'))

  def test_2010_C(self):
    '''MNStorage.update(): Attempt to update an object with existing PID, raises IdentifierNotUnique.
    '''
    new_pid = 'update_object_pid_1'
    old_pid = 'fitch2.mc'
    sci_obj, sys_meta = self.generate_test_object(new_pid)
    client = d1_client.mnclient.MemberNodeClient(self.options.gmn_url)
    self.assertRaises(d1_common.types.exceptions.IdentifierNotUnique,
      client.update, old_pid, StringIO.StringIO(sci_obj), new_pid, sys_meta,
      vendorSpecific=self.include_subjects('test_user_1'))

  # ----------------------------------------------------------------------------
  # MNStorage.delete()
  # ----------------------------------------------------------------------------

  def test_2100(self):
    '''MNStorage.delete()
    '''
    client = d1_client.mnclient.MemberNodeClient(self.options.gmn_url)
    # Find the PID for a random object that exists on the server.
    pid = self.find_valid_pid(client)
    # Delete the object on GMN.
    pid_deleted = client.delete(pid, vendorSpecific=self.include_subjects(
      gmn_test_client.GMN_TEST_SUBJECT_TRUSTED))
    self.assertEqual(pid, pid_deleted.value())
    # Verify that the object no longer exists.
    self.assertRaises(d1_common.types.exceptions.DataONEException, client.describe, pid)

  # ----------------------------------------------------------------------------
  # MNStorage.archive()
  # ----------------------------------------------------------------------------

  def test_2200_A(self):
    '''MNStorage.archive()
    '''
    client = d1_client.mnclient.MemberNodeClient(self.options.gmn_url)
    # Find the PID for a random object that exists on the server.
    pid = self.find_valid_pid(client)
    # Archive the object on GMN.
    pid_archived = client.archive(pid, vendorSpecific=self.include_subjects('test_user_1'))
    self.assertEqual(pid, pid_archived.value())
    # Verify that the object no longer exists.
    self.assertRaises(d1_common.types.exceptions.DataONEException, client.describe, pid)

  # ----------------------------------------------------------------------------
  # Unicode.
  # ----------------------------------------------------------------------------

  def test_2300(self):
    '''Unicode: GMN and libraries handle Unicode correctly.
    '''
    # Many of these do not work when running against the Django development
    # server. This is due to a bug in the development server. Disabled until I
    # add logic to detect development server and skip the identifiers that the
    # development server cannot handle.
    print 'Unicode tests currently disabled. See code for details'
    return
    client = d1_client.mnclient.MemberNodeClient(self.options.gmn_url)
#    test_doc_path = os.path.join(self.options.int_path,
#                                 'src', 'test', 'resources', 'd1_testdocs',
#                                 'encodingTestSet')
#    test_ascii_strings_path = os.path.join(test_doc_path,
#                                           'testAsciiStrings.utf8.txt')
    test_ascii_strings_path = './tricky_identifiers_unicode.txt'
    file_obj = codecs.open(test_ascii_strings_path, 'rb', 'utf-8')
    for line in file_obj:
      line = line.strip()
      try:
        pid_unescaped, pid_escaped = line.split('\t')
      except ValueError:
        continue
      scidata, sysmeta = self.generate_test_object(pid_unescaped)
      # Create the object on GMN.
      client.create(pid_unescaped, StringIO.StringIO(scidata), sysmeta,
        vendorSpecific=self.include_subjects(gmn_test_client.GMN_TEST_SUBJECT_TRUSTED))
      # Retrieve the object from GMN.
      scidata_retrieved = client.get(pid_unescaped,
        vendorSpecific=self.include_subjects(gmn_test_client.GMN_TEST_SUBJECT_TRUSTED))\
          .read()
      sysmeta_obj_retrieved = client.getSystemMetadata(pid_unescaped,
        vendorSpecific=self.include_subjects(gmn_test_client.GMN_TEST_SUBJECT_TRUSTED))
      # Round-trip validation.
      self.assertEqual(scidata_retrieved, scidata)
      self.assertEqual(sysmeta_obj_retrieved.identifier.value()\
                       .encode('utf-8'), scidata)


# ==============================================================================

def main():
  log_setup()

  # Command line options.
  parser = optparse.OptionParser()
  parser.add_option('--d1-root', dest='d1_root', action='store', type='string', default='http://0.0.0.0:8000/cn/') # default=d1_common.const.URL_DATAONE_ROOT
  parser.add_option('--gmn-url', dest='gmn_url', action='store', type='string', default='http://0.0.0.0:8000')
  parser.add_option('--gmn2-url', dest='gmn2_url', action='store', type='string', default='http://0.0.0.0:8001/')
  parser.add_option('--gmn-replicate-src-ref', dest='replicate_src_ref', action='store', type='string', default='gmn_dryad')
  parser.add_option('--cn-url', dest='cn_url', action='store', type='string', default='http://cn.dataone.org/cn/')
  parser.add_option('--xsd-path', dest='xsd_url', action='store', type='string', default='http://129.24.0.11/systemmetadata.xsd')
  parser.add_option('--obj-path', dest='obj_path', action='store', type='string', default='./test_objects')
  parser.add_option('--obj-url', dest='obj_url', action='store', type='string', default='http://localhost/test_objects/')
  parser.add_option('--wrapped', action='store_true', default=False, dest='wrapped')
  parser.add_option('--verbose', action='store_true', default=False, dest='verbose')
  parser.add_option('--quick', action='store_true', default=False, dest='quick')
  parser.add_option('--test', action='store', default='', dest='test', help='run a single test')
#  parser.add_option('--unicode-path', dest='unicode_path', action='store', type='string', default='/home/dahl/D1/svn/allsoftware/cicore/d1_integration/src/test/resources/d1_testdocs/encodingTestSet/testUnicodeStrings.utf8.txt')
  parser.add_option('--integration-path', dest='int_path', action='store', type='string', default='./d1_integration')
  parser.add_option('--debug', action='store_true', default=False, dest='debug')

  (options, args) = parser.parse_args()

  if not options.verbose:
    logging.getLogger('').setLevel(logging.ERROR)

  s = TestSequenceFunctions
  s.options = options

  if options.test != '':
    suite = unittest2.TestSuite(map(s, [options.test]))
    #suite.debug()
  else:
    suite = unittest2.TestLoader().loadTestsFromTestCase(s)

#  if options.debug == True:
#    unittest2.TextTestRunner(verbosity=2).debug(suite)
#  else:
  unittest2.TextTestRunner(verbosity=2, failfast=True).run(suite)


if __name__ == '__main__':
  main()
