__author__ = 'michael'

import sys
import time
import logging
import threading
import hashlib

import redis
import zkclient
import continuum


cc_logger = logging.getLogger('cache_client')

DEFAULT_CACHE_PREFIX = '/cache/'
DEFAULT_ZK_ADDRESS = '127.0.0.1:2181'
PWD_SUFFIX = '6379'
SEPARATOR = ':'


class CacheAccess(object):
    def Get(self, key):
        pass

    def Set(self, key, value, expired_time):
        pass

    def Exists(self, key):
        pass

    def Strlen(self, key):
        pass

    def Delete(self, key):
        pass


class CachException(Exception):
    pass


class CacheClient(zkclient.NodeChildrenListener, CacheAccess):
    """
    usage:
        cache_client = CacheClient(your_ns, your_biz, cipher, zk_address)
        DBA should offer you :your_ns, your_biz, cipher, zk_address

        When you call cache_client.Destory(), cache_client is not available!

        logger:
            your should config cache_client logger, LOG_LEVEL=INFO

    """

    def __init__(self, ns, biz, cipher='', zk_address=DEFAULT_ZK_ADDRESS):
        super(CacheClient, self).__init__(DEFAULT_CACHE_PREFIX + ns)
        self.ns = ns
        self.biz = biz
        self.cipher = cipher
        self.zk_address = zk_address
        self.continuum = continuum.Continuum(self.biz)
        self.cc_lock = threading.Lock()
        self.redis_dict = {} # string:Redis
        self.zk_client = zkclient.ZkClient(self.zk_address, '/tmp/zk.log')
        self.zk_client.AddNodeChildrenListener(self)
        self.destroyed = False
        children = None
        try:
            children = self.zk_client.GetChildren(self.get_znode_path(), True)
        except Exception, e:
            cc_logger.error(self._Description(e.message))
        cc_logger.debug('self.get_znode_path():%s' % self.get_znode_path())
        cc_logger.debug('chidlren:%s' % children)
        if children:
            self.Update(children)

        cc_logger.debug('CacheClient(%s,%s) started' % (self.ns, self.biz))

    def __get_redis_pwd(key):
        source = '%s%s' % (key, PWD_SUFFIX)
        md = hashlib.md5()
        md.update(source)
        result = md.hexdigest()
        return '%c%c%c%c%c%c%c%c' % (
            result[5], result[2], result[6], result[8], result[15], result[12], result[16], result[18])


    def _Description(self, msg):
        return '[ns:%s, biz:%s, msg:%s]' % (self.ns, self.biz, msg)

    def Update(self, children_name_list):
        self.cc_lock.acquire()
        try:
            if self.destroyed:
                cc_logger.debug(self._Description('CacheClient.Update() self has been destroyed'))
                return True
            if not children_name_list:
                cc_logger.warn(self._Description('CacheClient.Update() empty children_name_list'))
                return True

            new_continuum = continuum.Continuum(self.biz)
            new_redis_dict = {}
            for child in children_name_list:
                try:
                    sp = child.split(':')
                    cc_logger.debug('sp:%s' % sp)
                    nickname = sp[0]
                    host = sp[1]
                    port = int(sp[2])
                    pwd = sp[3]
                    timeout = int(sp[4])
                    r_pwd = self.cipher
                    if pwd != '':
                        r_pwd = self.__get_redis_pwd(pwd)

                    connection_kwargs = {
                        'host': host,
                        'port': port,
                        'db': 0,
                        'password': r_pwd,
                        'socket_timeout': timeout
                    }
                    redis_blocking_connection_pool = redis.BlockingConnectionPool(**connection_kwargs)
                    redis_client = redis.Redis(connection_pool=redis_blocking_connection_pool)
                    new_redis_dict[nickname] = redis_client
                    new_continuum.Add(nickname, 100)
                except Exception, e:
                    cc_logger.error(self._Description(e.message))

            if new_redis_dict:
                # rehash
                self.continuum.Clear()
                self.continuum = new_continuum
                self.continuum.Rebuild()
                # release old redis resources!
                for (k, v) in self.redis_dict.iteritems():
                    try:
                        if hasattr(v, 'connection_pool'):
                            v.connection_pool.disconnect()
                    except Exception, e:
                        cc_logger.error(self._Description(e))
                self.redis_dict.clear()
                # assemble new redis_dict
                self.redis_dict = new_redis_dict

            cc_logger.debug(self.redis_dict)
        finally:
            self.cc_lock.release()

    def _GenerateCacheKey(self, prefix, name):
        return '%s%s%s' % (prefix, SEPARATOR, name)

    def _LocateRedis(self, key):
        nickname = self.continuum.Locate(continuum.Continuum.Hash(key))
        with self.cc_lock:
            return self.redis_dict.get(nickname) # never raise KeyError

    def Get(self, key):
        cache_key = self._GenerateCacheKey(self.biz, key)
        client = self._LocateRedis(key)
        if not client:
            cc_logger.error(self._Description('CacheClient.Get(%s) _LocateRedis return None' % key))
            raise CachException(self._Description('CacheClient.Get(%s) not available Redis.' % key))
        return client.get(cache_key)

    def Set(self, key, value, expired_time=0):
        cache_key = self._GenerateCacheKey(self.biz, key)
        client = self._LocateRedis(key)
        if not client:
            cc_logger.error(self._Description('CacheClient.Get(%s) _LocateRedis return None' % key))
            raise CachException(self._Description('CacheClient.Get(%s) not available Redis.' % key))
        return client.set(cache_key, value, ex=expired_time)


    def Delete(self, key):
        cache_key = self._GenerateCacheKey(self.biz, key)
        client = self._LocateRedis(key)
        if not client:
            cc_logger.error(self._Description('CacheClient.Delete(%s) _LocateRedis return None' % key))
            raise CachException(self._Description('CacheClient.Delete(%s) not available Redis.' % key))
        return client.delete(cache_key)

    def Exists(self, key):
        cache_key = self._GenerateCacheKey(self.biz, key)
        client = self._LocateRedis(key)
        if not client:
            cc_logger.error(self._Description('CacheClient.Exists(%s) _LocateRedis return None' % key))
            raise CachException(self._Description('CacheClient.Exists(%s) not available Redis.' % key))
        return client.exists(cache_key)


    def Strlen(self, key):
        cache_key = self._GenerateCacheKey(self.biz, key)
        client = self._LocateRedis(key)
        if not client:
            cc_logger.error(self._Description('CacheClient.Strlen(%s) _LocateRedis return None' % key))
            raise CachException(self._Description('CacheClient.Strlen(%s) not available Redis.' % key))
        return client.strlen(cache_key)

    def Destory(self):
        self.cc_lock.acquire()
        try:
            for (k, v) in self.redis_dict.iteritems():
                try:
                    if hasattr(v, 'connection_pool'):
                        v.connection_pool.disconnect()
                except Exception, e:
                    cc_logger.error(self._Description(e))
            self.redis_dict.clear()
            self.continuum.Clear()
            self.destroyed = True
        finally:
            self.cc_lock.release()


if __name__ == '__main__':
    # zk1 = zkclient.ZkClient("127.0.0.1:2181")
    # zk2 = zkclient.ZkClient("127.0.0.1:2181")
    LOG_FORMAT = "[%(process)d:%(thread)d:%(levelname)s:%(asctime)s:%(filename)s(%(lineno)d):%(name)s] %(message)s"
    logging.basicConfig(level=logging.DEBUG, stream=sys.stdout, format=LOG_FORMAT, datefmt='%Y%m%d%H%M%S')
    cc = CacheClient('namespace', 'test')
    k = 'mykey1'
    print cc.Set(k, 'HI', 100)
    print cc.Get(k)

    time.sleep(5)
    print 'destory cc'
    cc.Destory()
    # print cc.Exists(k)
    # print cc.Strlen(k)
    # print cc.Delete(k)
    # print cc.Exists(k)
    # del cc
    # print 'cc deleted'
    time.sleep(3)
