# Copyright (c) Twisted Matrix Laboratories
# Copyright (c) Alexander Walters
# See LICENSE for details.

import struct
import socket
import types
import datetime

from twisted.internet import reactor, protocol, defer
from twisted.words.protocols import irc
from twisted.persisted import styles
from twisted.names import client
from twister.python.versions import Version as _Version

version = _Version('dccclient', 0, 1, 1)

NUL = chr(0)
CR = chr(015)
NL = chr(012)
LF = NL
SPC = chr(040)


def is_ip(address):
    """
    Returns True if ``address`` is an IP address recognized by the socket
    module
    """
    try:
        socket.inet_aton(address)
    except socket.errno:
        return False
    return True


class IRCClient(irc.IRCClient):
    # USERHOST Methods and attributes, used to get the IP address as seen by
    # the IRC server (A wort of the NAT age)
    _userhost_defer = {}
    _userhost_cache = {}
    _host_cache = {}

    def _nslookup(self, address):
        """
        Caching DNS lookup of ``address``
        returns a deferred with the IP address as a value
        """
        if is_ip(address):
            # If it's already just an IP address, just give it back
            d = defer.Deferred()
            d.callback(address)
            return d
        elif address in self._host_cache:
            # If we looked it up before, just return the old answer
            # XXX: Should we expire the cache?
            d = defer.Deferred()
            d.callback(self._host_cache[address])
            return d
        else:
            # Conveniently, twisted.names returns deferreds.
            return client.getHostByName(self.host_remote.strip())

    def userhost(self, user):
        """
        Looks up the user's host mask via the IRC server.
        Returns a deferred with the host portion (after the @) as a value
        """
        user = user.lower()
        d = defer.Deferred()
        if user in self._userhost_cache:
            d.callback(self._userhost_cache[user])
        else:
            self._userhost_defer[user] = d
            self.sendLine('USERHOST {0}'.format(user))
        return d

    def irc_RPL_USERHOST(self, user, params):
        # This method name would start with an underscore if it could
        nick, _, _ = params[-1].partition('=')
        nick = nick.lower()
        if nick in self._userhost_defer:
            _, _, host = params[-1].partition('@')
            self._userhost_defer[nick].callback(host)
            del self._userhost_defer[nick]
            self._userhost_cache[nick] = host

    # DCC SEND and (outbound) RESUME support
    dcc_sends = {}

    @defer.inlineCallbacks
    def dccSend(self, user, file):
        """
        Initiates a DCC Send to user.
        file is either a file name or a file-like object (.read and .close
        methods)
        """
        # if file is a string, assume it is a file name.
        if isinstance(file, types.StringType):
            file = open(file, 'r')

        # get file size and file name
        size = irc.fileSize(file)
        name = getattr(file, "name", "file@%s" % (id(file),))

        # What IP am I bound to for my connection to irc? thats the one i want
        # to listen on for dcc
        interface = self.transport.getHost()

        # Build the factory, giving it the file object and a deferred we will
        # use later.  Factory will callback the deferred with a protocol
        # instance
        d = defer.Deferred()
        factory = DccSendFactory(file, d)
        port = reactor.listenTCP(0, factory, 1, interface.host)

        # I need the port number I am listening on.  Shove deferred into a dict
        # keyed by that port so we can later get the protocol instance by port
        # number
        address = port.getHost()
        self.dcc_sends[address.port] = d

        # I need to tell the other end to connect to me on an IP.  Sadly, in a
        # world of NAT, I can't know what IP they should connect to.  If only
        # we had a server or something to tell us what our outside IP was?
        host_remote = yield self._userhost(self.nickname)
        ip_remote = yield self._nslookup(host_remote.strip)

        # For some reason DCC sends IP addresses as longs.  *shrug*
        my_address = struct.unpack('!L', socket.inet_aton(ip_remote))[0]

        # All done!  Tell other side to connect now.
        args = ['SEND', name, str(my_address), str(address.port)]
        if size is not None:
            args.append(str(size))
        args = ' '.join(args)
        self.ctcpMakeQuery(user, [('DCC', args)])

    @defer.inlineCallbacks
    def dccAcceptResume(self, user, fileName, port, resumePos):
        # Yank the protocol out of the dict (set by dccSend).  Is the port
        # already an int?
        proto = yield self.dcc_sends[int(port)]

        # Tell the protocol to seek to a file position
        proto.setStart(resumePos)

        # tell the other side we are ready for them to connect
        self.ctcpMakeQuery(user, [
            ('DCC', ['ACCEPT', fileName, port, resumePos])
        ])

    def dccDoResume(self, user, fileName, port, resumePos):
        # Someone asked us if we would resume.  We only should if we have an
        # outbound file transfer waiting.
        if int(port) in self.dcc_sends:
            self.dccAcceptResume(user, fileName, port, resumePos)

    # These are some fixes unrelated to DCC
    def ctcpQuery_SOURCE(self, user, channel, data):
        if self.sourceURL:
            nick = user.split("!")[0]
            self.ctcpMakeReply(nick, [('SOURCE', self.sourceURL)])

    def ctcpQuery_TIME(self, user, channel, data):
        nick = user.split("!")[0]
        self.ctcpMakeReply(nick,
                           [('TIME', datetime.datetime.now().ctime())])


class DccSendProtocol(protocol.Protocol, styles.Ephemeral):
    """Protocol for an outgoing Direct Client Connection SEND.
    """

    blocksize = 1024
    file = None
    bytesSent = 0
    completed = 0
    connected = 0

    def __init__(self, file):
        # file should already be a file(-like) object, but just in case.
        if isinstance(file, types.StringType):
            self.file = open(file, 'r')
        else:
            self.file = file

    def connectionMade(self):
        # Since transfers are one per customer, close the connection if we
        # already started with someone else.
        if self.factory.used:
            self.transport.loseConnection()
        else:
            self.factory.used = True
            self.connected = 1
            # start sending!
            self.sendBlock()

    def setStart(self, position):
        # used for resume.  Seeks to file position, but only if we havn't
        # already started the transfer
        if not self.connected:
            self.file.seek(position)

    def dataReceived(self, data):
        bytesShesGot = struct.unpack("!I", data)[0]
        if bytesShesGot < self.bytesSent:
            # If you got less than I gave, I'm going to wait until you tell me
            # you got everything I gave
            return
        elif bytesShesGot > self.bytesSent:
            # If you got more than I gave...
            self.transport.loseConnection()
            return
        # otherwise, just send it already
        self.sendBlock()

    def sendBlock(self):
        block = self.file.read(self.blocksize)
        if block:
            self.transport.write(block)
            self.bytesSent = self.bytesSent + len(block)
        else:
            self.transport.loseConnection()
            self.completed = 1

    def connectionLost(self, reason):
        self.connected = 0
        if hasattr(self.file, "close"):
            self.file.close()


class DccSendFactory(protocol.Factory):
    protocol = DccSendProtocol

    def __init__(self, file, deferred):
        self.file = file
        self.d = deferred

        # Protocol will check this to see if this port was already connected
        # to.  Only one per customer, sorry!
        self.used = False

    def buildProtocol(self, connection):
        p = self.protocol(self.file)
        p.factory = self

        # this puts the protocol in the outbound sends dictionary in the bot.
        self.d.callback(p)
        return p
