# -*- coding: utf-8 -*-
# test_predictSNR.py
# This module provides the tests for the predictSNR function.
# Copyright 2014 Giuseppe Venturini
# This file is part of python-deltasigma.
#
# python-deltasigma is a 1:1 Python replacement of Richard Schreier's 
# MATLAB delta sigma toolbox (aka "delsigma"), upon which it is heavily based.
# The delta sigma toolbox is (c) 2009, Richard Schreier.
#
# python-deltasigma is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# LICENSE file for the licensing terms.

"""This module provides the test class for the predictSNR() function.
"""
from __future__ import division

import unittest
import numpy as np
import deltasigma as ds

class TestPredictSNR(unittest.TestCase):
    """Test class for predictSNR()"""

    def setUp(self):
        pass

    def test_predictSNR_1(self):
        """Test function for predictSNR() 1/3"""
        snr_ref = [-32.3447, -22.3447, -12.3447, -2.3447, 7.6553, 17.6553,
                   27.6553, 37.6552, 47.6545, 57.6475, 67.5768, 72.4043,
                   76.8266, 77.5913, 78.2773, 78.8451, 79.2116, 79.0974,
                   -np.Inf, -np.Inf, -np.Inf, -np.Inf, -np.Inf]
        amp_ref = [-120., -110., -100., -90., -80., -70., -60., -50., -40.,
                   -30, -20, -15, -10, -9, -8, -7, -6, -5, -4, -3, -2, -1,
                   0]
        k0_ref = [1.6289, 1.6289, 1.6289, 1.6289, 1.6289, 1.6289,
                  1.6289, 1.6289, 1.6288, 1.6283, 1.6227, 1.6088,
                  1.5596, 1.5383, 1.5088, 1.4663, 1.4003, 1.2747,
                  0., 0., 0., 0., 0.]
        k1_ref = [1.6289, 1.6289, 1.6289, 1.6289, 1.6289, 1.6289,
                  1.6289, 1.6289, 1.6287, 1.6274, 1.6142, 1.5819,
                  1.4752, 1.4326, 1.3768, 1.3025, 1.1995, 1.0387,
                  0.3706, 0., 0., 0., 0.]
        sigma_e2_ref = [0.3634, 0.3634, 0.3634, 0.3634, 0.3634, 0.3634,
                   0.3634, 0.3634, 0.3634, 0.3634, 0.3634, 0.3631,
                   0.3607, 0.3591, 0.3566, 0.3525, 0.3459, 0.3352,
                   0.3178, 0., 0., 0., 0.]
        order = 5
        osr = 32
        f0 = 0
        Hinf = 1.5
        ntf = ds.synthesizeNTF(order, osr, 2, Hinf, f0)
        snr_pred, amp_pred, k0, k1, sigma_e2 = ds.predictSNR(ntf, osr, None, f0)

        # Delete garbage data
        for check in (np.isinf, np.isnan):
            i = check(snr_ref)
            snr_ref = np.delete(snr_ref, np.where(i))
            amp_ref = np.delete(amp_ref, np.where(i))
            i = check(snr_pred)
            snr_pred = np.delete(snr_pred, np.where(i))
            amp_pred = np.delete(amp_pred, np.where(i))

        self.assertTrue(np.allclose(snr_pred, snr_ref, atol=1e-2, rtol=1e-3))
        self.assertTrue(np.allclose(amp_pred, amp_ref, atol=1e-2, rtol=5e-4))
        self.assertTrue(np.allclose(k0, k0_ref, atol=1e-3, rtol=1e-2))
        self.assertTrue(np.allclose(k1, k1_ref, atol=1e-3, rtol=50e-2))
        self.assertTrue(np.allclose(sigma_e2, sigma_e2_ref, atol=1e-3, rtol=1e-2))

    def test_predictSNR_2(self):
        """Test function for predictSNR() 2/2"""
        snr_ref = [-54.7270, -53.5149, -52.3028, -51.0907, -49.8786, -48.6664,
                   -47.4543, -46.2422, -45.0301, -43.8180, -42.6058, -41.3937,
                   -40.1816, -38.9695, -37.7574, -36.5452, -35.3331, -34.1210,
                   -32.9089, -31.6967, -30.4846, -29.2725, -28.0604, -26.8483,
                   -25.6361, -24.4240, -23.2119, -21.9998, -20.7877, -19.5755,
                   -18.3634, -17.1513, -15.9392, -14.7270, -13.5149, -12.3028,
                   -11.0907, -9.8786, -8.6664, -7.4543, -6.2422, -5.0301,
                   -3.8180, -2.6058, -1.3937, -0.1816, 1.0305, 2.2426,
                   +3.4548, 4.6669, 5.8790, 7.0911, 8.3032, 9.5154,
                   +10.7275, 11.9396, 13.1517, 14.3638, 15.5759, 16.7881,
                   +18.0002, 19.2123, 20.4244, 21.6365, 22.8485, 24.0606,
                   +25.2727, 26.4847, 27.6967, 28.9087, 30.1206, 31.3324,
                   +32.5442, 33.7558, 34.9673, 36.1785, 37.3895, 38.6002,
                   +39.8103, 41.0198, 42.2285, 43.4360, 44.6421, 45.8462,
                   +47.0478, 48.2458, 49.4393, 50.6266, 51.8058, 52.9741,
                   +54.1277, 55.2617, 56.3694, 57.4405, 58.4607, 59.4074,
                   +60.2442, 60.9031, 61.2360, 60.8103]
        amp_ref = [-120.0000, -118.7879, -117.5758, -116.3636, -115.1515, -113.9394,
                   -112.7273, -111.5152, -110.3030, -109.0909, -107.8788, -106.6667,
                   -105.4545, -104.2424, -103.0303, -101.8182, -100.6061, -99.3939,
                    -98.1818, -96.9697, -95.7576, -94.5455, -93.3333, -92.1212,
                    -90.9091, -89.6970, -88.4848, -87.2727, -86.0606, -84.8485,
                    -83.6364, -82.4242, -81.2121, -80.0000, -78.7879, -77.5758,
                    -76.3636, -75.1515, -73.9394, -72.7273, -71.5152, -70.3030,
                    -69.0909, -67.8788, -66.6667, -65.4545, -64.2424, -63.0303,
                    -61.8182, -60.6061, -59.3939, -58.1818, -56.9697, -55.7576,
                    -54.5455, -53.3333, -52.1212, -50.9091, -49.6970, -48.4848,
                    -47.2727, -46.0606, -44.8485, -43.6364, -42.4242, -41.2121,
                    -40.0000, -38.7879, -37.5758, -36.3636, -35.1515, -33.9394,
                    -32.7273, -31.5152, -30.3030, -29.0909, -27.8788, -26.6667,
                    -25.4545, -24.2424, -23.0303, -21.8182, -20.6061, -19.3939,
                    -18.1818, -16.9697, -15.7576, -14.5455, -13.3333, -12.1212,
                    -10.9091, -9.6970, -8.4848, -7.2727, -6.0606, -4.8485,
                     -3.6364, -2.4242, -1.2121, 0.]
        k0_ref = [3.6301, 3.6301, 3.6301, 3.6301, 3.6301, 3.6301,
                  3.6301, 3.6301, 3.6301, 3.6301, 3.6301, 3.6301,
                  3.6301, 3.6301, 3.6301, 3.6301, 3.6301, 3.6301,
                  3.6301, 3.6301, 3.6301, 3.6301, 3.6301, 3.6301,
                  3.6301, 3.6301, 3.6301, 3.6301, 3.6301, 3.6301,
                  3.6301, 3.6301, 3.6301, 3.6301, 3.6301, 3.6301,
                  3.6301, 3.6301, 3.6301, 3.6301, 3.6301, 3.6301,
                  3.6301, 3.6301, 3.6301, 3.6301, 3.6301, 3.6301,
                  3.6301, 3.6301, 3.6301, 3.6301, 3.6301, 3.6301,
                  3.6301, 3.6301, 3.6301, 3.6301, 3.6301, 3.6301,
                  3.6301, 3.6301, 3.6301, 3.6301, 3.6301, 3.6301,
                  3.6300, 3.6300, 3.6300, 3.6300, 3.6299, 3.6299,
                  3.6298, 3.6298, 3.6297, 3.6295, 3.6293, 3.6291,
                  3.6287, 3.6283, 3.6277, 3.6270, 3.6260, 3.6246,
                  3.6228, 3.6205, 3.6173, 3.6131, 3.6075, 3.6000,
                  3.5899, 3.5762, 3.5576, 3.5320, 3.4961, 3.4447,
                  3.3690, 3.2518, 3.0562, 2.6817]
        k1_ref = [3.6301, 3.6301, 3.6301, 3.6301, 3.6301, 3.6301,
                  3.6301, 3.6301, 3.6301, 3.6301, 3.6301, 3.6301,
                  3.6301, 3.6301, 3.6301, 3.6301, 3.6301, 3.6301,
                  3.6301, 3.6301, 3.6301, 3.6301, 3.6301, 3.6301,
                  3.6301, 3.6301, 3.6301, 3.6301, 3.6301, 3.6301,
                  3.6301, 3.6301, 3.6301, 3.6301, 3.6301, 3.6301,
                  3.6301, 3.6301, 3.6301, 3.6301, 3.6301, 3.6301,
                  3.6301, 3.6301, 3.6301, 3.6301, 3.6301, 3.6301,
                  3.6301, 3.6301, 3.6301, 3.6301, 3.6301, 3.6301,
                  3.6301, 3.6301, 3.6301, 3.6301, 3.6301, 3.6301,
                  3.6301, 3.6301, 3.6301, 3.6300, 3.6300, 3.6300,
                  3.6300, 3.6299, 3.6299, 3.6298, 3.6297, 3.6296,
                  3.6295, 3.6293, 3.6290, 3.6286, 3.6282, 3.6275,
                  3.6267, 3.6256, 3.6242, 3.6223, 3.6198, 3.6164,
                  3.6120, 3.6062, 3.5984, 3.5881, 3.5744, 3.5561,
                  3.5318, 3.4993, 3.4557, 3.3969, 3.3171, 3.2075,
                  3.0547, 2.8367, 2.5142, 2.0040]
        sigma_e2_ref = [0.3634, 0.3634, 0.3634, 0.3634, 0.3634, 0.3634,
                        0.3634, 0.3634, 0.3634, 0.3634, 0.3634, 0.3634,
                        0.3634, 0.3634, 0.3634, 0.3634, 0.3634, 0.3634,
                        0.3634, 0.3634, 0.3634, 0.3634, 0.3634, 0.3634,
                        0.3634, 0.3634, 0.3634, 0.3634, 0.3634, 0.3634,
                        0.3634, 0.3634, 0.3634, 0.3634, 0.3634, 0.3634,
                        0.3634, 0.3634, 0.3634, 0.3634, 0.3634, 0.3634,
                        0.3634, 0.3634, 0.3634, 0.3634, 0.3634, 0.3634,
                        0.3634, 0.3634, 0.3634, 0.3634, 0.3634, 0.3634,
                        0.3634, 0.3634, 0.3634, 0.3634, 0.3634, 0.3634,
                        0.3634, 0.3634, 0.3634, 0.3634, 0.3634, 0.3634,
                        0.3634, 0.3634, 0.3634, 0.3634, 0.3634, 0.3634,
                        0.3634, 0.3634, 0.3634, 0.3634, 0.3634, 0.3634,
                        0.3634, 0.3634, 0.3634, 0.3634, 0.3634, 0.3634,
                        0.3634, 0.3633, 0.3633, 0.3633, 0.3633, 0.3632,
                        0.3630, 0.3628, 0.3624, 0.3616, 0.3603, 0.3579,
                        0.3536, 0.3459, 0.3319, 0.3059]

        amp = np.linspace(-120, 0, 100)
        order = 4
        osr = 64
        f0 = 0.333
        Hinf = 1.2

        ntf = ds.synthesizeNTF(order, osr, 2, Hinf, f0)
        snr_pred, amp_pred, k0, k1, sigma_e2 = ds.predictSNR(ntf, osr, amp, f0)

        # Delete garbage data
        for check in (np.isinf, np.isnan):
            i = check(snr_ref)
            snr_ref = np.delete(snr_ref, np.where(i))
            amp_ref = np.delete(amp_ref, np.where(i))
            i = check(snr_pred)
            snr_pred = np.delete(snr_pred, np.where(i))
            amp_pred = np.delete(amp_pred, np.where(i))

        self.assertTrue(np.allclose(snr_pred, snr_ref, atol=1e-2, rtol=1e-3))
        self.assertTrue(np.allclose(amp_pred, amp_ref, atol=1e-2, rtol=5e-4))
        self.assertTrue(np.allclose(k0, k0_ref, atol=1e-3, rtol=1e-2))
        self.assertTrue(np.allclose(k1, k1_ref, atol=1e-3, rtol=50e-2))
        self.assertTrue(np.allclose(sigma_e2, sigma_e2_ref, atol=1e-3, rtol=1e-2))

