
import math

import design, shape, profile

from nesoni import config, legion

def decorate(prof, pos, align, amount=0.2):
    amount = prof(pos)*amount
    pos += amount * align
    deco_profile = profile.Profile(
        [ pos+amount*i for i in [-1,0,1]],
        [ amount*i      for i in [0,1,0] ],
    )    
    return prof + deco_profile.clipped(prof.start(),prof.end())

@config.Float_flag('mill_diameter', 'Diameter of milling bit.')
@config.Bool_flag('mill_ball', 'Is it a ball-mill rather than an end-mill? Extra depth will be added so the bit cuts through into the base.')
class Miller(config.Configurable):
    mill_diameter = 3.0
    mill_ball = True
        
    def miller(self, pattern):    
        extent = pattern.extent()
        height = extent.zmax - extent.zmin
        
        bit_pad = self.mill_diameter * 1.25
        extra_depth = self.mill_diameter*0.5 if self.mill_ball else 0.0
        a = bit_pad
        b = bit_pad + (height+extra_depth)/10.0
        pad_cone = shape.extrude_profile(profile.Profile(
                [extent.zmin-extra_depth,extent.zmax],[a*2,b*2]
            ), 
            cross_section=lambda d: shape.circle(d,16))
        
        extra = b + 0.5
        
        extent = pattern.extent()
        mill = shape.block(
            extent.xmin-extra,       extent.xmax+extra,
            extent.ymin-extra,       extent.ymax+extra,
            extent.zmin-extra_depth, extent.zmax,
        )
        
        mill.remove( pattern.polygon_mask().to_3().minkowski_sum(pad_cone) )
        mill.add( pattern )
        return mill


@config.Bool_flag('draft', 'Draft output')
class Make_base(config.Action):
    draft = False
    
    # We'll need a lot of memory, so run exclusively
    def cores_required(self):
        return legion.coordinator().get_cores()

    def save(self, shape, prefix):
        prefix = self.get_workspace()/prefix
        
        shape.save(prefix + '.stl')
        
        import sketch
        sketch.sketch(shape, prefix + '-sketch.svg')

    def _before_run(self):
        if self.draft:
            shape.draft_mode()
        self.get_workspace()

class Make(config.Action_with_output_dir, Make_base):
    pass


class Working(object): pass

@config.String_flag('prefix', 'Output file prefix (useful if you want to make your design several different ways).')
@config.Float_flag('gap', 
    'Printing: Amount of gap all around in the joins between segments. '
    'The best value for this will depend on the accuracy of your printer. '
    'If a joint is too loose and leaks, it can be sealed using wax or similar.')
class Make_instrument(config.Action_with_working_dir, Make_base):
    gap = 0.35
    prefix = ''

    def _before_run(self):
        self.working = Working()
        self.working.designer = design.load(self.working_dir)
        self.working.spec = self.working.designer.instrument

    def _after_run(self):
        del self.working
        
    def save(self, shape, prefix):
        Make_base.save(self, shape, self.prefix+prefix)

    def make_instrument(
             self,
             inner_profile, outer_profile, 
             hole_positions, hole_diameters, hole_vert_angles, hole_horiz_angles,
             xpad, ypad, with_fingerpad):
        outside = shape.extrude_profile(outer_profile)
        instrument = outside.copy()        
        bore = shape.extrude_profile(inner_profile)
                      
        for i, pos in enumerate(hole_positions):
            angle = hole_vert_angles[i]
            radians = angle*math.pi/180.0
        
            height = outer_profile(pos)*0.5 
            inside_height = inner_profile(pos)*0.5 
            shift = math.sin(radians) * height
            
            #hole_length = (
            #    math.sqrt(height*height+shift*shift) + 
            #    hole_diameters[i]*0.5*abs(math.sin(radians)) + 
            #    4.0)
            
            hole_diameter_correction = math.cos(radians) ** -0.5
            hole_diameter = hole_diameters[i] * hole_diameter_correction
                
            cross_section = shape.squared_circle(xpad[i], ypad[i]).with_effective_diameter
            
            h1 = inside_height*0.75
            shift1 = math.sin(radians)*h1
            h2 = height*1.5
            shift2 = math.sin(radians)*h2
            hole = shape.extrusion([h1,h2],
                [ cross_section(hole_diameter).offset(0.0,shift1),
                  cross_section(hole_diameter).offset(0.0,shift2) ])
            hole.rotate(1,0,0, -90)
                
            #hole = shape.prism(
            #    hole_length, hole_diameters[i],
            #    cross_section)                
            #hole.rotate(1,0,0, -90-angle)
            
            hole.rotate(0,0,1, hole_horiz_angles[i])
            hole.move(0,0,pos + shift)
            
            if with_fingerpad[i]:
                pad_height = height*0.5+0.5*math.sqrt(height**2-(hole_diameters[i]*0.5)**2)
                pad_depth = (pad_height-inside_height)
                pad_mid = pad_depth*0.25
                pad_diam = hole_diameter*1.3
                fingerpad = shape.extrude_profile(
                    profile.Profile([-pad_depth,-pad_mid,0.0],[pad_diam+pad_mid*2,pad_diam+pad_mid*2,pad_diam]),
                    cross_section=cross_section)
                fingerpad_negative = shape.extrude_profile(
                    profile.Profile([0.0,pad_mid],[pad_diam,pad_diam+pad_mid*8]),
                    cross_section=cross_section)
                
                wall_angle = -math.atan2(0.5*(outer_profile(pos+pad_diam*0.5)-outer_profile(pos-pad_diam*0.5)),pad_diam)*180.0/math.pi
                fingerpad.rotate(1,0,0, wall_angle)
                fingerpad_negative.rotate(1,0,0, wall_angle)
                
                fingerpad.move(0,0,pad_height)
                fingerpad_negative.move(0,0,pad_height)
                fingerpad.rotate(1,0,0, -90)
                fingerpad_negative.rotate(1,0,0, -90)
                fingerpad.rotate(0,0,1, hole_horiz_angles[i])
                fingerpad_negative.rotate(0,0,1, hole_horiz_angles[i])
                fingerpad.move(0,0,pos) # - 0.2*hole_diameters[i]*math.sin(radians)) #????????????????????
                fingerpad_negative.move(0,0,pos) # - 0.2*hole_diameters[i]*math.sin(radians)) #?????????????????????
                outside.add(fingerpad)
                outside.remove(fingerpad_negative)
                instrument.add(fingerpad)
                instrument.remove(fingerpad_negative)

            bore.add(hole)
            if angle or hole_horiz_angles[i]:
                outside.remove(hole)
        
        instrument.remove(bore)
        instrument.rotate(0,0,1, 180)
        
        self.working.instrument = instrument
        self.working.outside = outside
        self.working.bore = bore        
        self.save(instrument, 'instrument')

    def segment(self, cuts, length, up=False):
        working = self.working
        remainder = working.instrument.copy()
        inner = working.spec.inner
        outer = working.spec.outer
        
        if up:
            cuts = [ length-item for item in cuts[::-1] ]
            remainder.rotate(0,1,0, 180)
            remainder.move(0,0,length)
            inner = inner.reversed().moved(length)
            outer = outer.reversed().moved(length)
        
        shapes = [ ]
        for cut in cuts:
            d1 = inner(cut)
            d3 = outer(cut)
            d2 = (d1+d3) / 2.0
            
            
            d4 = outer.maximum() * 2.0
            p1 = cut-d2*0.4
            p3 = cut+d2*0.4
            p2 = p1 + (d3-d1)*0.25
        
            #0.4mm clearance all around
            # radius -> diameter    * 2
            # adjust both diamerers / 2
            # net effect            * 1
            # 0.4mm was loose
            
            d1a = d1 - self.gap
            p1b = p1 - self.gap
            
            d2a = d2 - self.gap
            d2b = d2 + self.gap
            
            prof_inside = profile.Profile(
                [ p1,  p2,  p3,  length+50 ],
                [ d1a, d2a, d2a, d4 ],
                [ d1a, d2a, d4,  d4 ],
            )
            prof_outside = profile.Profile(
                [ p1b, p2,  p3,  length+50 ],
                [ d1,  d2b, d2b, d4 ],
                [ d1,  d2b, d4,  d4 ],
            )
            mask_inside = shape.extrude_profile(prof_inside)
            mask_outside = shape.extrude_profile(prof_outside)
            
            item = remainder.copy()
            item.remove(mask_outside)                
            remainder.clip(mask_inside)
            shapes.append(item)
        shapes.append(remainder)
        shapes = shapes[::-1]
        
        for i, item in enumerate(shapes):
            item.rotate(0,1,0, 180)
            self.save(item, 'segment-%d-of-%d' % (i+1,len(shapes)))
        
        self.working.segments = shapes

        
        
