from deployer.contrib.nodes.apt_get import AptGet
from deployer.contrib.nodes.config import Config
from deployer.contrib.nodes.virtualenv import VirtualEnv
from deployer.query import Q
from deployer.node import SimpleNode, required_property
from deployer.utils import esc1

from pygments.lexers import BashLexer


init_d_template = """\
#!/bin/bash

### BEGIN INIT INFO
# Provides:          uwsgi-%(slug)s
# Required-Start:    $remote_fs $syslog $all
# Required-Stop:
# Default-Start:     2 3 4 5
# Default-Stop:      0 1 6
# Description:       Uwsgi process for web server. (Script generated by the Deployer)
# ### END INIT INFO


do_start() {
    # Only start if PID file does not exists or PID inside is invalid.
    if [ ! -f %(pidfile)s ] || ! kill -0 ` cat '%(pidfile)s'  ` 2> /dev/null; then
        %(start_command)s
    else
        echo "Already running..."
    fi
}

do_reload() {
    %(reload_command)s
}

do_stop() {
    %(stop_command)s
}

case  "$1" in
    start)
        do_start
        ;;
    restart|reload|force-reload)
        do_reload
        ;;
    stop)
        do_stop
        ;;
    *)
        echo "Usage $0 start|reload" >&2
        exit 3
        ;;
esac
"""


class Uwsgi(SimpleNode):
    virtual_env_location = required_property()
    slug = required_property()
    socket = 'localhost:3032' # Can be either a tcp socket or unix file socket
    wsgi_module = required_property()
    run_from_directory = required_property()
    threads = 10
    workers = 2
    username = required_property()

    uwsgi_download_url = 'http://projects.unbit.it/downloads/uwsgi-1.9.12.tar.gz'

    # HTTP
    use_http = False
    http_port = 80

    class _packages(AptGet):
        # libxml2-dev is required for compiling uwsgi
        packages = ('libxml2-dev',)


    class virtual_env(VirtualEnv):
        @property
        def requirements(self):
            return (
                self.parent.uwsgi_download_url,

                # For monitoring uwsgi.
                'uwsgitop',
                )
        virtual_env_location = Q.parent.virtual_env_location

    def setup(self):
        self._packages.install()
        self.virtual_env.upgrade_requirements()
        self.init_d_file.setup()

    @property
    def pidfile(self):
        return '/tmp/django-uwsgi-%s.pid' % self.slug

    @property
    def logfile(self):
        return '/tmp/uwsgi-log-%s' % self.slug

    @property
    def stats_socket(self):
        return '/tmp/uwsgi-stats-%s' % self.slug

    def _get_start_command(self, daemonize=True):
        """
        UWSGI startup command
        Because of --daemonize, we don't need upstart anymore.
        """
        if self.use_http:
            socket = '--http=0.0.0.0:%s' % self.http_port
        else:
            socket = '--socket=%s' % self.socket

        return '\\\n   '.join([
                    '%(virtual_env)s/bin/uwsgi',
                                    '--chdir=%(chdir)s',
                                    '--module=%(wsgi_module)s',
                                    '%(uwsgi_socket)s',
                                    '--threads=%(threads)i',
                                    '--processes=%(workers)i',
                                    '--stats=%(stats)s',
                                    '%(pidfile)s',
                                    '--uid=%(username)s',
                                    '--chmod-socket=%(chmod_socket)s',
                                    '--home=%(virtual_env)s',
                                    '%(daemonize)s',
                    ]) % {
                                    'chdir': self.run_from_directory,
                                    'virtual_env': self.virtual_env_location,
                                    'uwsgi_socket': socket,
                                    'pidfile': ('--pidfile=%s' % self.pidfile if daemonize else ''),
                                    'daemonize': '--daemonize=%s' % self.logfile if daemonize else '',
                                    'wsgi_module': self.wsgi_module,
                                    'threads': self.threads,
                                    'workers': self.workers,
                                    'stats': self.stats_socket,
                                    'username': self.username,
                                    'chmod_socket': 666,
                    }

    @property
    def reload_command(self):
        return "kill -SIGHUP ` cat '%s' ` " % esc1(self.pidfile)

    @property
    def stop_command(self):
        return "kill -SIGQUIT ` cat '%s' ` && rm '%s' " % (esc1(self.pidfile), esc1(self.pidfile))

    def monitor_log(self):
        """
        Show uwsgi tail.
        """
        self.hosts.sudo("tail -f '%s' " % esc1(self.logfile), ignore_exit_status=True)

    def top(self):
        """
        Run uwsgi top
        """
        with self.hosts.prefix(self.virtual_env.activate_cmd):
            self.hosts.sudo("uwsgitop '%s'" % self.stats_socket)

    def start(self, daemonize=True):
        """
        Start uWSGI stack
        """
        for h in self.hosts:
            if not h.exists(self.pidfile):
                h.sudo(self._get_start_command(daemonize))
            else:
                print 'Pidfile %s already exists' % self.pidfile

    def run_in_shell(self):
        self.start(daemonize=False)

    def stop(self):
        """
        Kill all the uWSGI stack.
        """
        self.hosts.sudo(self.stop_command)

    def status(self):
        """
        True when this uwsgi process is running
        """
        for h in self.hosts:
            if h.exists(self.pidfile):
                print 'Running'

    def reload(self):
        """
        Reload (gracefully) all the workers and the master process.
        """
        self.hosts.sudo(self.reload_command)

    def rmpidfile(self):
        """
        Remove pidfile, sometimes it can happen that the pidfile was created, and the
        server crached due to a bad configuration, without removing the pidfile.
        """
        if self.console.confirm('Remove pidfile'):
            self.hosts.sudo("kill -SIGQUIT ` cat '%s' ` || rm '%s' " % (esc1(self.pidfile), esc1(self.pidfile)))


    class init_d_file(Config):
        @property
        def slug(self):
            return 'uwsgi-%s' % self.parent.slug

        @property
        def remote_path(self):
            return '/etc/init.d/%s' % self.slug

        lexer = BashLexer
        use_sudo = True
        make_executable = True

        @property
        def content(self):
            self = self.parent

            return init_d_template % {
                    'slug': self.slug,
                    'run_from_directory': self.run_from_directory,
                    'pidfile': self.pidfile,
                    'start_command': self._get_start_command(True),
                    'reload_command': self.reload_command,
                    'stop_command': self.stop_command,
                    }

        def setup(self):
            Config.setup(self)

            # Automatically start on system boot.
            self.hosts.sudo("update-rc.d '%s' defaults" % self.slug)
