from digue.lilypond_language import *
import digue.instruments_settings as settings

#################
### Functions ###
#################

def _int_to_string(n):
	if n == 1:
		return 'One'
	elif n == 2:
		return 'Two'
	elif n == 3:
		return 'Three'
	elif n == 4:
		return 'Four'
	elif n == 5:
		return 'Five'
	elif n == 6:
		return 'Six'
	elif n == 7:
		return 'Seven'
	elif n == 8:
		return 'Eight'
	else:
		raise NameError('Number is too big.')

def _no_midi():
	return '\set Staff.midiMaximumVolume = #0.0\n'

def _double_angle_brackets(s):
	return "<<\n"+s+">>\n"

def _score_expr(s):
	t = "\n"+s if len(s.splitlines()) > 1 else " "+s+" "
	return "\\tag #'score {"+t+"}\n"

def _part_expr(s):
	return "\\tag #'part {\n"+s+"}\n"

###############
### Objects ###
###############

class _Transposition:

	def __init__(self, transposition):
		self.__transposition = transposition

	def __get_transposition_string(self, config):
		# self.__transposition: "c" or "bes" or "bes,"...
		# Note
		tsp = language[config["language"]][self.__transposition[0]]
		# Accidentals
		cursor = 1
		if 'es' in self.__transposition or 'is' in self.__transposition:
			tsp += language[config["language"]][self.__transposition[1:3]]
			cursor = 3
		# Octave
		tsp += self.__transposition[cursor:len(self.__transposition)]
		return tsp

	def string(self, config, content):
		transposition = self.__get_transposition_string(config)
		do = language[config["language"]]["c"]
		return "\\transposition {0}\n" \
		       "\\transpose {0} {1}' <<\n{2}>>\n".format(
		             transposition, do, content)

class _InstrumentName:

	def __init__(self, instrument_name, short_instrument_name, number):
		self.__instrument_name = instrument_name
		self.__short_instrument_name = short_instrument_name
		self.__number = number

	def string(self, config, context="Staff"):
		number = "" if self.__number == [0] else " " + str(self.__number[0])
		s = ""
		if config["display_instruments_names"] == "1":
			s += '\set {0}.instrumentName = "{1}"\n'.format(
					context, self.__instrument_name+number)
		if config["display_short_instruments_names"] == "1":
			tmp = '\set {0}.shortInstrumentName = "{1}"'.format(
					context, self.__short_instrument_name+number)
			s += _score_expr(tmp)
		return s

class _MIDIInstrument:

	def __init__(self, midi_instrument):
		self.__midi_instrument = midi_instrument

	def string(self):
		return '\set Staff.midiInstrument = "{0}"\n'.format(
		           self.__midi_instrument)

class _StringTunings:

	def __init__(self, string_tunings):
		self.__string_tunings = string_tunings

	def string(self):
		return '\set Staff.stringTunings = #{0}\n'.format(
		            self.__string_tunings)

class _Clef:

	def __init__(self, clef):
		self.__clef               = clef
		self.__clef_concert_pitch = clef

	@property
	def concert_pitch(self):
		return self.__clef_concert_pitch

	@concert_pitch.setter
	def concert_pitch(self, clef_concert_pitch):
		self.__clef_concert_pitch = clef_concert_pitch

	def string(self):
		return '\clef "{0}"\n'.format(self.__clef)

	def string_concert_pitch(self):
		return '\clef "{0}"\n'.format(self.__clef_concert_pitch)

	def pitch(self):
		if self.__clef_concert_pitch == "treble":
			return "''"
		elif self.__clef_concert_pitch == "alto" or \
		     self.__clef_concert_pitch == "treble_8":
			return "'"
		elif self.__clef_concert_pitch == "bass":
			return ""
		elif self.__clef_concert_pitch == "bass_8":
			return ","
		else:
			raise NameError('Unknown clef: {0}'.format(
				self.__clef_concert_pitch))

class _Lyrics:

	def __init__(self, variable, number):
		self.__variable = variable
		self.__number = number

	def _variable(self, variable):
		return '{0} = \lyricmode {{\nta\n}}\n\n'.format(variable)

	def variables(self):
		if self.__number == 0:
			return ""
		var = self.__variable + "_Lyrics"
		s = ""
		if self.__number == 1:
			s += self._variable(var) + "\n"
			return s
		for i in range(1, self.__number+1):
			var_tmp = var + _int_to_string(i)
			s += self._variable(var_tmp)
		s += "\n"
		return s

	def _new(self, variable, lyricsto):
		return '\\new Lyrics = "{0}" \lyricsto "{1}" {{ \\{0} }}\n'.format(
		           variable, lyricsto)

	def staff(self):
		s = ""
		if self.__number == 0:
			return ""
		var = self.__variable + "_Lyrics"
		lyricsto = self.__variable + "_VoiceOne"
		if self.__number == 1:
			s += self._new(var, lyricsto)
			return s
		for i in range(1, self.__number+1):
			var_tmp = var + _int_to_string(i)
			s += self._new(var_tmp, lyricsto)
		return s

class _Context:
	"""Intermediate-level contexts - staves.

	See http://lilypond.org/doc/v2.19/Documentation/notation/contexts-explained#intermediate_002dlevel-contexts-_002d-staves
	"""

	def __init__(self, variable, number, type):
		self.__variable = variable
		self.__number = number
		self.__type = type

	def __order(self, i):
		"""Return the number to use in \\voiceX command for voice i.
		
		See "Voice order" section at
		http://lilypond.org/doc/v2.18/Documentation/notation/multiple-voices#single_002dstaff-polyphony
		"""
		if i == 1:
			return 1
		elif i == self.__number:
			return 2
		elif i == 2:
			return 3
		else:
			return 4

	def __var(self, var, type, voice_number, example):
		return '{0} = {1} {{\n{2}{3}\n}}\n\n'.format(
		           var, type, voice_number, example)

	def variables(self, type, example):
		"""Return variables definitions of context's voices."""
		s = ""
		if self.__number == 1:
			voice_number = ""
			var = self.__variable + "_VoiceOne"
			s += self.__var(var, type, voice_number, example)
			return s + "\n"
		for i in range(1, self.__number+1):
			var = self.__variable + "_Voice" + _int_to_string(i)
			voice_number = '\\voice' + _int_to_string(self.__order(i)) + "\n"
			s += self.__var(var, type, voice_number, example)
		s += "\n"
		return s

	def staff(self, content):
		"""Return the staff part of the context."""
		s = '\\new {0}Staff = "{1}" '.format(self.__type, self.__variable)
		s += _double_angle_brackets(content)
		return s

	def __new(self, var):
		return '\\new {0}Voice = "{1}" << \global {{ \\{1} }} >>\n'.format(
		           self.__type, var)

	def voices(self):
		"""Return instantiation of context's voices."""
		s = ""
		for i in range(1, self.__number+1):
			var = self.__variable + "_Voice" + _int_to_string(i)
			s += self.__new(var)
		return s

class _StaffContext(_Context):

	def __init__(self, variable, voices_number, clef):
		super().__init__(variable, voices_number, "")
		self.__clef = clef

	def variables(self, config):
		do = language[config["language"]]["c"]
		return super().variables("\\relative " + do + self.__clef.pitch(), do)

class _TabContext(_Context):

	def __init__(self, variable, voice_number):
		return super().__init__(variable, voice_number, "Tab")

class _DrumContext(_Context):

	def __init__(self, variable, voice_number):
		super().__init__(variable, voice_number, "Drum")

	def variables(self):
		return super().variables("\drummode", "hh")

class _GroupedStaves:

	def __init__(self, type, variable):
		self.__type = type
		self.__variable = variable

	def staff(self, s):
		return '\\new {0} = "{1}" '.format(self.__type, self.__variable) \
		     + _double_angle_brackets(s)

class _PianoStaff(_GroupedStaves):

	def __init__(self, variable):
		super().__init__("PianoStaff", variable)

class _StaffGroup(_GroupedStaves):

	def __init__(self, variable):
		super().__init__("StaffGroup", variable)

###########################
### Instruments Classes ###
###########################

class _Instrument:

	def __init__(self, variable, instrument_name, short_instrument_name):
		self.__number = [0]
		self._instrument_name = _InstrumentName(
				instrument_name, short_instrument_name, self.number)
		self.__variable = variable

	@property
	def variable(self):
		if self.__number == [0]:
			return self.__variable
		else:
			return self.__variable + _int_to_string(self.__number[0])

	@property
	def number(self):
		return self.__number

	@number.setter
	def number(self, n):
		self.__number[0] = n

	def part(self, config):
		s = '\include "../Infos.ly"\n' \
		    '\include "../Notes/{0}.ily"\n\n' \
		    "music = \keepWithTag #'part <<\n\\{0}\n>>\n\n" \
		    '\score {{\n\music\n\layout {{ }}\n}}\n\n' \
		    "\score {{\n\\unfoldRepeats \\articulate << \music >>\n" \
		    '\midi {{ \midiTempo }}\n}}\n'.format(self.variable)
		return indent(s, config)

class StaffInstrument(_Instrument):

	SETTINGS = [
			settings.Variable,
			settings.InstrumentName,
			settings.ShortInstrumentName,
			settings.MIDIInstrument,
			settings.Clef,
			settings.NumberOfVoices
	]

	def __init__(self, variable, instrument_name, short_instrument_name, 
	             midi_instrument, clef, voices_number):
		super().__init__(variable, instrument_name, short_instrument_name)
		self._clef = _Clef(clef)
		self._midi_instrument = _MIDIInstrument(midi_instrument)
		self._staff = _StaffContext(self.variable,
		                            int(voices_number),
		                            self._clef)

	def notes(self, config):
		tmp = self._instrument_name.string(config) \
		    + self._midi_instrument.string() \
		    + self._clef.string() \
		    + self._staff.voices()
		s = self._staff.variables(config) \
		  + self.variable + ' = ' + self._staff.staff(tmp)
		return indent(s, config)

class TransposingInstrument(StaffInstrument):

	SETTINGS = [
			settings.Variable,
			settings.InstrumentName,
			settings.ShortInstrumentName,
			settings.MIDIInstrument,
			settings.Clef,
			settings.Transposition,
			settings.ClefConcertPitch,
			settings.NumberOfVoices,
			settings.ConcertPitch
	]

	def __init__(self, variable, instrument_name, short_instrument_name,
	             midi_instrument, clef, transposition, clef_concert_pitch,
	             voices_number, concert_pitch_in_score):
		super().__init__(variable, instrument_name, short_instrument_name,
		                 midi_instrument, clef, voices_number)
		self._transposition = _Transposition(transposition)
		self._clef.concert_pitch = clef_concert_pitch
		self._concert_pitch_in_score = int(concert_pitch_in_score)

	def _notes_score_written_pitch(self, config):
		tmp = self._instrument_name.string(config) \
		    + self._midi_instrument.string() \
		    + self._clef.string() \
		    + self._transposition.string(config, self._staff.voices())
		s = self._staff.variables(config) \
		  + self.variable + ' = ' + self._staff.staff(tmp)
		return indent(s, config)

	def _notes_score_concert_pitch(self, config):
		var = self.variable+"_Voices"
		v = "\\" + var + "\n"
		tmp = self._clef.string() + self._transposition.string(config, v)
		part = _part_expr(tmp)
		tmp = self._clef.string_concert_pitch() + v
		score = _score_expr(tmp)
		tmp = self._instrument_name.string(config) \
		    + self._midi_instrument.string() \
		    + part \
		    + score
		s = self._staff.variables(config) \
		  + var + ' = ' + _double_angle_brackets(self._staff.voices()) + "\n" \
		  + self.variable + ' = ' + self._staff.staff(tmp)
		return indent(s, config)

	def notes(self, config):
		if self._concert_pitch_in_score == 1:
			return self._notes_score_concert_pitch(config)
		else:
			return self._notes_score_written_pitch(config)

class DrumStaffInstrument(_Instrument):

	SETTINGS = [
			settings.Variable,
			settings.InstrumentName,
			settings.ShortInstrumentName,
			settings.NumberOfVoices,
	]

	def __init__(self, variable, instrument_name, short_instrument_name,
	             voices_number):
		super().__init__(variable, instrument_name, short_instrument_name)
		self._staff = _DrumContext(self.variable, int(voices_number))

	def notes(self, config):
		tmp = self._instrument_name.string(config) \
		    + self._staff.voices()
		s = self._staff.variables() \
		  + self.variable + ' = ' + self._staff.staff(tmp)
		return indent(s, config)

class PianoStaffInstrument(_Instrument):

	SETTINGS = [
			settings.Variable,
			settings.InstrumentName,
			settings.ShortInstrumentName,
			settings.MIDIInstrument,
			settings.NumberOfVoicesRight,
			settings.NumberOfVoicesLeft
	]

	def __init__(self, variable, instrument_name, short_instrument_name,
	             midi_instrument, voices_number_right, voices_number_left):
		super().__init__(variable, instrument_name, short_instrument_name)

		self._midi_instrument = _MIDIInstrument(midi_instrument)

		self._clef_right = _Clef("treble")
		self._clef_left  = _Clef("bass")

		self._staff_right = _StaffContext(self.variable+"_Right",
		                                  int(voices_number_right),
		                                  self._clef_right)
		self._staff_left  = _StaffContext(self.variable+"_Left",
		                                  int(voices_number_left),
		                                  self._clef_left)

		self._piano_staff = _PianoStaff(self.variable)

	def notes(self, config):
		tmp = self._midi_instrument.string() \
		    + self._clef_right.string() \
		    + self._staff_right.voices()
		staff_right = self._staff_right.staff(tmp)

		tmp = self._midi_instrument.string() \
		    + self._clef_left.string() \
		    + self._staff_left.voices()
		staff_left = self._staff_left.staff(tmp)

		tmp = self._instrument_name.string(config, context="PianoStaff") \
		    + staff_right + staff_left
		piano_staff = self._piano_staff.staff(tmp)

		s = self._staff_right.variables(config) \
		  + self._staff_left.variables(config) \
		  + self.variable + ' = ' + piano_staff
		return indent(s, config)

class TabStaffInstrument(StaffInstrument):

	SETTINGS = [
			settings.Variable,
			settings.InstrumentName,
			settings.ShortInstrumentName,
			settings.MIDIInstrument,
			settings.Clef,
			settings.IsTransposing,
			settings.StringTunings,
			settings.NumberOfVoices,
			settings.ScoreTabDisplay
	]

	def __init__(self, variable, instrument_name, short_instrument_name,
	             midi_instrument, clef, is_transposing, string_tunings,
	             voices_number, display):
		super().__init__(variable, instrument_name, short_instrument_name,
		                 midi_instrument, clef, voices_number)
		self._string_tunings = _StringTunings(string_tunings)
		self._display = display
		self._tabstaff = _TabContext(self.variable, int(voices_number))
		self._staffgroup = _StaffGroup(self.variable)
		self.__is_transposing = int(is_transposing)
		if self.__is_transposing == 1:
			self._transposition = _Transposition("c")
			self._clef.concert_pitch = "bass"

	def _notes_score(self, config):
		if self.__is_transposing == 1:
			voices = self._transposition.string(config, self._staff.voices())
		else:
			voices = self._staff.voices()
		tmp = self._instrument_name.string(config) \
		    + self._midi_instrument.string() \
		    + self._string_tunings.string() \
		    + self._clef.string() \
		    + voices
		s = self._staff.variables(config) \
		  + self.variable + ' = ' + self._staff.staff(tmp)
		return indent(s, config)

	def _notes_tab(self, config):
		tmp = self._instrument_name.string(config, context="TabStaff") \
		    + self._midi_instrument.string() \
		    + self._string_tunings.string() \
		    + "\\tabFullNotation\n" \
		    + self._tabstaff.voices()
		s = self._staff.variables(config) \
		  + self.variable + ' = ' + self._tabstaff.staff(tmp)
		return indent(s, config)

	def _notes_score_tab(self, config):
		if self.__is_transposing == 1:
			voices = self._transposition.string(config, self._staff.voices())
		else:
			voices = self._staff.voices()
		tmp = self._midi_instrument.string() \
		    + self._clef.string() \
		    + voices
		score = self._staff.staff(tmp)

		tmp = _no_midi() \
		    + self._string_tunings.string() \
		    + self._tabstaff.voices()
		tab = self._tabstaff.staff(tmp)

		tmp = self._instrument_name.string(config, context="StaffGroup") \
		    + score + tab
		s = self._staff.variables(config) \
		  + self.variable + ' = ' + self._staffgroup.staff(tmp)
		return indent(s, config)

	def notes(self, config):
		if self._display == "Score":
			return self._notes_score(config)
		if self._display == "Tab":
			return self._notes_tab(config)
		else:
			return self._notes_score_tab(config)

class LyricsStaffInstrument(StaffInstrument):

	SETTINGS = [
			settings.Variable,
			settings.InstrumentName,
			settings.ShortInstrumentName,
			settings.MIDIInstrument,
			settings.Clef,
			settings.NumberOfVoices,
			settings.NumberOfVerses
	]

	def __init__(self, variable, instrument_name, short_instrument_name,
	             midi_instrument, clef, voices_number, verses_number):
		super().__init__(variable, instrument_name, short_instrument_name,
		                 midi_instrument, clef, voices_number)
		self._lyrics = _Lyrics(self.variable, int(verses_number))

	def notes(self, config):
		tmp = self._instrument_name.string(config) \
		    + self._midi_instrument.string() \
		    + self._clef.string() \
		    + self._staff.voices() \
		    + self._lyrics.staff()
		s = self._staff.variables(config) \
		  + self._lyrics.variables() \
		  + self.variable + ' = ' + self._staff.staff(tmp)
		return indent(s, config)

types = {
		"StaffInstrument": StaffInstrument,
		"TransposingInstrument": TransposingInstrument,
		"PianoStaffInstrument": PianoStaffInstrument,
		"DrumStaffInstrument": DrumStaffInstrument,
		"TabStaffInstrument": TabStaffInstrument,
		"LyricsStaffInstrument": LyricsStaffInstrument,
}
