import digue.digue as di
import digue.instruments_classes as ic
import digue.predefined_instruments as pi
import digue.general_settings as gs
import digue.instruments_settings as settings

import os
import uuid
import collections
import configparser
import appdirs

from tkinter import *
from tkinter import ttk
from tkinter import filedialog

class _ChooseDir:

	def __choose_project_dir(self):
		project_dir = filedialog.asksaveasfilename(initialfile="MyProject")
		if project_dir != "" and project_dir != ():
			self.__project_dir.set(project_dir)

	def __init__(self, master):
		self.__project_dir = StringVar()
		home = os.path.expanduser("~")
		self.__project_dir.set(os.path.join(home,"MyProject"))
		ttk.Label(master, text="Project directory") \
				.grid(column=0, row=0, sticky=W, padx=0)
		ttk.Label(master, textvariable=self.__project_dir) \
				.grid(column=1, row=0, sticky=E, padx=5)
		ttk.Button(master, text="Browse", command= \
				lambda: self.__choose_project_dir()) \
				.grid(column=2, row=0, sticky=E, padx=0)

	def get_project_dir(self):
		return self.__project_dir.get()

class _ConfigOptions:

	def __init__(self, master):
		self.__settings = []
		i=0
		for setting in gs.settings_list:
			frame = ttk.Frame(master)
			self.__settings.append(setting(frame))
			frame.grid(column=0, row=i, pady=1, stick=(E, W))
			frame.columnconfigure(0, weight=1)
			i += 1
		ttk.Separator(master).grid(column=0, columnspan=2, row=i, stick=(N,S,E,W), pady=10)

	def get_general_settings(self):
		settings = {}
		for setting in self.__settings:
			settings[setting.id] = setting.get()
		gs.save(settings)
		return settings

class _Instrument:

	def __init__(self, master, iid, predefined_settings):
		self.iid = iid
		self.__type_string = predefined_settings["type"]
		self.__type = ic.types[self.__type_string]

		self.__label = Label(master, text="Instrument settings")
		self.__type_frame = ttk.Frame(master)
		Label(self.__type_frame, text="Type") \
				.grid(column=0, row=0, sticky=W)
		Label(self.__type_frame, text=self.__type_string) \
				.grid(column=0, row=1, sticky=W)
		self.__settings_frame = ttk.Frame(master)
		self.__canvas = Canvas(master, width=200, height=1)

		self.settings = settings.InstrumentSettings(
				self.__settings_frame, self.__type.SETTINGS)
		for setting_id, value in predefined_settings.items():
			self.settings.set(setting_id, value)

	def show_settings(self):
		self.__label.grid(column=0, row=0, pady=10)
		self.__type_frame.grid(column=0, row=1, pady=2, sticky=W)
		self.__settings_frame.grid(column=0, row=2, sticky=(E, W))
		self.__settings_frame.columnconfigure(0, weight=1)
		self.__canvas.grid(column=0, row=3)

	def get_dict(self):
		dict = self.settings.get_dict()
		dict["type"] = self.__type_string
		return dict

	def create(self):
		return self.__type(*self.settings.get_list())

class _InstrumentsTrees:

	def __update_selected_instruments_tree(self):
		# Save current selection:
		t = self.__selected_instruments_tree
		selection = t.selection()
		# Delete all entries:
		t.delete(*t.get_children())
		# Display all entries:
		for ins in self.__selected_instruments_list:
			name = ins.settings.get("instrument_name")
			if name == "":
				name = ins.settings.get("variable")
			if name == "":
				name = "Unnamed instrument"
			t.insert('', 'end', text=name, tags=ins.iid, iid=ins.iid)
			t.tag_bind(ins.iid, '<1>',
					lambda e, arg=ins: self.__instrument_params(arg))
			# Restore previous selection:
			if ins.iid in selection:
				t.selection_add(ins.iid)

	def __move(self, n):
		t = self.__selected_instruments_tree
		l = self.__selected_instruments_list
		selection = t.selection() if n <= 0 else t.selection()[::-1]
		if len(selection) < 1:
			return
		for item_id in selection:
			index = t.index(item_id)
			if index+n < 0 or index+n > len(l)-1:
				return
			l[index], l[index+n] = l[index+n], l[index]
		self.__update_selected_instruments_tree()

	def __move_up(self):
		self.__move(-1)

	def __move_down(self):
		self.__move(+1)

	def __move_more(self, n):
		t = self.__selected_instruments_tree
		l = self.__selected_instruments_list
		selection = t.selection() if n == 0 else t.selection()[::-1]
		if len(selection) < 1:
			return
		for item_id in selection:
			index = t.index(item_id)
			l[index], l[n] = l[n], l[index]
		self.__update_selected_instruments_tree()

	def __move_on_top(self):
		self.__move_more(0)

	def __move_at_bottom(self):
		self.__move_more(len(self.__selected_instruments_list)-1)

	def __delete_instrument(self):
		selection = self.__selected_instruments_tree.selection()
		if len(selection) < 1:
			return
		for instrument in self.__selected_instruments_list[::-1]:
			if instrument.iid in selection:
				self.__selected_instruments_list.remove(instrument)
		if self.__current_instrument != None:
			for widget in self.__settings_frame.winfo_children():
				widget.grid_forget()
		self.__current_instrument = None
		self.__update_selected_instruments_tree()

	def __add_instrument(self, predefined_settings):
		iid = str(uuid.uuid1())
		instrument = _Instrument(
				self.__settings_frame, iid, predefined_settings)
		self.__selected_instruments_list.append(instrument)
		self.__update_selected_instruments_tree()

	def __instrument_params(self, instrument):
		self.__update_selected_instruments_tree()
		if self.__current_instrument != None:
			for widget in self.__settings_frame.winfo_children():
				widget.grid_forget()
		instrument.show_settings()
		self.__current_instrument = instrument

	def __available_instruments_frame_init(self, m):
		s = ttk.Scrollbar(m)
		t = ttk.Treeview(m, yscroll=s.set)
		s.config(command=t.yview)

		instruments_by_categories = collections.OrderedDict()
		for instrument in pi.predefined_instruments:
			category = instrument["category"]
			if category not in instruments_by_categories.keys():
				instruments_by_categories[category] = []
			instruments_by_categories[category].append(instrument)

		for category, instrument_list in instruments_by_categories.items():
			id = t.insert('', 'end', text=category)
			for instrument in instrument_list:
				name = instrument["instrument_name"]
				uuid_ = str(uuid.uuid1())
				t.insert(id, 'end', text=name, tags=uuid_)
				t.tag_bind(uuid_, '<Double-1>', \
						lambda e, arg=instrument: self.__add_instrument(arg))

		t.heading("#0", text="Available instruments")
		t.grid(column=0, row=0, sticky=(N, S, E, W))
		s.grid(column=1, row=0, sticky=(N, S, E, W))

	def __selected_instruments_frame_init(self, m):
		s = ttk.Scrollbar(m)
		t = self.__selected_instruments_tree = ttk.Treeview(m, yscroll=s.set)
		s.config(command=t.yview)
		t.heading("#0", text="Selected instruments")
		t.grid(column=0, row=0, sticky=(N, E, S, W))
		s.grid(column=1, row=0, sticky=(N, S, E, W))

	def __buttons_frame_init(self, master):
		ttk.Button(master, text="↟", width=1, command= \
				lambda: self.__move_on_top()).grid(column=0, row=0)
		ttk.Button(master, text="↑", width=1, command= \
				lambda: self.__move_up()).grid(column=0, row=1)
		ttk.Button(master, text="↓", width=1, command= \
				lambda: self.__move_down()).grid(column=0, row=2)
		ttk.Button(master, text="↡", width=1, command= \
				lambda: self.__move_at_bottom()).grid(column=0, row=3)
		ttk.Button(master, text="×", width=1, command= \
				lambda: self.__delete_instrument()).grid(column=0, row=4)

	def __instruments_trees_frame_init(self, master):
		self.__selected_instruments_list = []

		# Available instruments frame
		f = ttk.Frame(master)
		self.__available_instruments_frame_init(f)
		f.rowconfigure(0, weight=1)
		f.grid(column=0, row=0, sticky=(N, S, E, W), padx=5)

		# Selected instruments frame
		f = ttk.Frame(master)
		self.__selected_instruments_frame_init(f)
		f.rowconfigure(0, weight=1)
		f.grid(column=1, row=0, sticky=(N, S, E, W), padx=5)

		# Buttons frame
		f = ttk.Frame(master)
		self.__buttons_frame_init(f)
		f.grid(column=2, row=0, padx=5)

		# Instrument settings frame
		f = self.__settings_frame = ttk.Frame(master)
		f.grid(column=3, row=0, sticky=(W, E, N), padx=5)
		self.__canvas = Canvas(f, width=200, height=1).grid()
		self.__current_instrument = None

	def __save_session(self, file=None):
		if file == None:
			file = filedialog.asksaveasfilename(defaultextension=".dig")
		config = configparser.ConfigParser()
		for instrument in self.__selected_instruments_list:
			config[instrument.iid] = instrument.get_dict()
		with open(file, 'w') as configfile:
			config.write(configfile)

	def __open_session(self, file=None):
		if file == None:
			file = filedialog.askopenfilename()
		config = configparser.ConfigParser()
		config.read(file)
		self.__selected_instruments_list = []
		for iid in config.sections():
			settings = config[iid]
			instrument = _Instrument(self.__settings_frame, iid, settings)
			self.__selected_instruments_list.append(instrument)
		self.__update_selected_instruments_tree()

	def __restore_session(self):
		file = os.path.join(self.__data_dir, "previous_session.dig")
		self.__open_session(file)

	def __session_frame_init(self, master):
		appname = "digue"
		self.__data_dir = appdirs.user_data_dir(appname)
		if not os.path.isdir(self.__data_dir):
			os.mkdir(self.__data_dir)

		ttk.Button(master, text="Open a session", command= \
				lambda: self.__open_session()) \
				.grid(column=0, row=0, sticky=W, padx=5)
		ttk.Button(master, text="Save current session", command= \
				lambda: self.__save_session()) \
				.grid(column=1, row=0, sticky=W, padx=5)
		ttk.Button(master, text="Restore last session", command= \
				lambda: self.__restore_session()) \
				.grid(column=2, row=0, sticky=W, padx=5)

	def __init__(self, instruments_trees_frame, session_frame):
		self.__instruments_trees_frame_init(instruments_trees_frame)
		self.__session_frame_init(session_frame)

	def get_instruments_list(self):
		file = os.path.join(self.__data_dir, "previous_session.dig")
		self.__save_session(file)

		instruments_list = []
		for instrument in self.__selected_instruments_list:
			if instrument.settings.get("variable") == "":
				return 0
		for instrument in self.__selected_instruments_list:
			instruments_list.append(instrument.create())
		return instruments_list

class _Application:

	def __create_project(self):
		project_dir = self.__choose_project_dir.get_project_dir()
		if os.path.isdir(project_dir):
			from tkinter import messagebox
			messagebox.showinfo(
					message='{0}: directory already exists.'.format(
						project_dir),
					icon="error")
			return

		settings         = self.__general_settings.get_general_settings()
		instruments_list = self.__instruments_trees.get_instruments_list()
		if instruments_list == 0: # <=> a variable equals ""
			from tkinter import messagebox
			messagebox.showinfo(
					message='Missing variable name.', icon="error")
			return
		if len(instruments_list) == 0:
			from tkinter import messagebox
			messagebox.showinfo(
					message='No selected instruments.', icon="error")
			return

		os.mkdir(project_dir)
		os.chdir(project_dir)
		di.create_project(instruments_list, settings)
		self.__root.destroy()

	def __init__(self):
		self.__root = Tk()
		self.__root.title("Digue")

		# Main frame
		f = mainframe = ttk.Frame(self.__root, padding="3 3 3 3")
		f.grid(column=0, row=0, sticky=(N, W, E, S))
		f.columnconfigure(0, weight=1)
		f.rowconfigure(0, weight=1)

		# Session
		row = 0
		f = session_frame = ttk.Frame(mainframe)
		f.grid(column=0, row=row, sticky=(N, W, E, S), padx=10, pady=5)

		# Choose dir
		row += 1
		f = ttk.Frame(mainframe)
		self.__choose_project_dir = _ChooseDir(f)
		f.grid(column=0, row=row, sticky=(N, W, E, S), padx=10, pady=5)
		f.columnconfigure(0, weight=1)
		f.rowconfigure(0, weight=1)

		# General settings frame
		row += 1
		f = ttk.Frame(mainframe)
		self.__general_settings = _ConfigOptions(f)
		f.grid(column=0, row=row, sticky=(N, W, E, S), padx=10)
		f.columnconfigure(0, weight=1)
		f.rowconfigure(0, weight=1)

		# Instruments frame
		row += 1
		f = ttk.Frame(mainframe)
		self.__instruments_trees = _InstrumentsTrees(f, session_frame)
		f.grid(column=0, row=row, sticky=(N, W, E, S))
		f.columnconfigure(0, weight=1)
		f.rowconfigure(0, weight=1)
		mainframe.rowconfigure(row, minsize=500)

		# Buttons frame
		row += 1
		f = ttk.Frame(mainframe)
		ttk.Button(f, text="Cancel", command=self.__root.destroy) \
				.grid(column=0, row=0, sticky=E, padx=5, pady=5)
		ttk.Button(f, text="Ok", command=lambda: self.__create_project()) \
				.grid(column=1, row=0, sticky=E, padx=5, pady=5)
		f.grid(column=0, row=row, sticky=(N, E, S))

		self.__root.mainloop()

def main():
	application = _Application()

if __name__ == '__main__':
	main()

