from distutils.core import Command
from distutils.errors import DistutilsOptionError, DistutilsFileError
from xml.dom.minidom import Document
from mimetypes import guess_type
import os.path

#List of all valid categories and subcategories.
#Taken from http://standards.freedesktop.org/menu-spec/latest/apa.html
#Will try to keep up-to-date, though I don't think the spec changes often.
registered_categories = {
    'AudioVideo':['Database','Midi','Mixer','Sequencer','Tuner','TV',
        'AudioVideoEditing','Player','Recorder','DiscBurning','Music'],
    'Audio':['HamRadio','Midi','Mixer','Sequencer','Tuner','AudioVideoEditing',
        'Player','Recorder'],
    'Video':['TV','AudioVideoEditing','Player','Recorder'],
    'Development':['Building','Debugger','IDE','GUIDesigner','Profiling',
        'RevisionControl','Translation','Database','ProjectManagement',
        'WebDevelopment'],
    'Education':['Art','Construction','Music','Languages','Science',
        'ArtificialIntelligence','Astronomy','Biology','Chemistry',
        'ComputerScience','DataVisualization','Economy','Electricity',
        'Geography','Geology','Geoscience','History','ImageProcessing',
        'Literature','Math','NumericalAnalysis','MedicalSoftware','Physics',
        'Robotics','Sports','ParallelComputing'],
    'Game':['ActionGame','AdventureGame','ArcadeGame','BoardGame','BlocksGame'
        'CardGame','KidsGame','LogicGame','RolePlaying','Simulation',
        'SportsGame','StrategyGame','Emulator'],
    'Graphics':['2DGraphics','VectorGraphics','RasterGraphics','3DGraphics',
        'Scanning','OCR','Photography','Publishing','Viewer'],
    'Network':['Email','Dialup','InstantMessaging','Chat','IRCClient',
        'FileTransfer','HamRadio','News','P2P','RemoteAccess','Telephony',
        'VideoConference','WebBrowser','WebDevelopment'],
    'Office':['Calendar','ContactManagement','Database','Dictionary','Chart',
        'Email','Finance','FlowChart','PDA','ProjectManagement','Presentation',
        'Spreadsheet','WordProcessor','Photography','Publishing','Viewer'],
    'Settings':['DesktopSettings','HardwareSettings','Printing',
        'PackageManager','Security','Accessibility'],
    'System':['Emulator','FileTools','FileManager','TerminalEmulator',
        'Filesystem','Monitor','Security'],
    'Utility':['TextTools','TelephonyTools','Archiving','Compression',
        'FileTools','Accessibility','Calculator','Clock','TextEditor'],
}


class gen_pxml(Command):

    user_options = [
        ('outfile=', None,
        'specify where to put the resulting xml data'),
        ('force', 'f',
        'overwrite outfile if it already exists'),

        #All subsequent user options set specific data fields in the PXML.
        #Most you will need can be autogenerated from the setup function, but
        #can be overridden with these options.  Some options, though, have no
        #corresponding setup function parameter and will need to be specified.
        ('id=', None,
        'specify the unique id to use for this application (default is based on name from setup function)'),
        ('appdata=', None,
        'specify the name of the appdata directory (default is same as id)'),
        ('title=', None,
        'specify a title (default is name from setup function)'),
        ('description=', None,
        'specify a description for the program (default is from setup function)'),
        ('exec-command=', None,
        'specify the command executed by the PND (default is first script listed in setup function; only one command is supported for now)'),
        ('exec-args=', None,
        'specify any arguments to be passed to the command specified by "exec"'),
        ('exec-startdir=', None,
        'specify the directory in which the command should be started (default is the root of the PND file)'),
        #('exec-nostandalone', None,
        #"activate this flag if the command should only be run with associations.  Since those aren't implemented yet, just don't use this flag"),
        ('exec-nobackground', None,
        'activate this flag if users should not be able to switch applications while this is running'),
        ('exec-nox', None,
        'activate this flag if the command should not be run in an X environment'),
        ('exec-xreq', None,
        'activate this flag if the command requires an X environment to run'),
        ('icon=', None,
        'specify an icon file to use'),
        ('info=', None,
        'specify a help file to use'),
        ('previewpics=', None,
        'specify any comma-separated preview pictures to use'),
        ('author=', None,
        'specify the author (default is from setup function)'),
        ('author-email=', None,
        "specify the author's email (default is from setup function)"),
        ('author-website=', None,
        "specify the author's website (default is url from setup function)"),
        ('version=', None,
        'specify the package version (default is from setup function)'),
        ('osversion=', None,
        'specify the required OS version, if any'),
        ('categories=', None,
        'specify categories in the form "Category1:Subcat1,Subcat2;Category2:Subcat3"'),
        #('associations=', None,
        #"I don't even think this is implemented in libpnd, so I won't have it here."),
        ('clockspeed=', None,
        'specify the required system clock, in Hz, if needed'),
        ('mkdir=', None,
        "specify comma-separated folders to create on the SD root, but don't use it, okay?"),
    ]


    def initialize_options(self):
        self.outfile = 'PXML.xml'
        self.force = False
        self.id = None
        self.appdata = None
        self.title = self.distribution.get_name()
        self.description = self.distribution.get_description()
        self.exec_command = None
        self.exec_args = None
        self.exec_startdir = None
        self.exec_nostandalone = False
        self.exec_nobackground = False
        self.exec_nox = False
        self.exec_xreq = False
        self.icon = None
        self.info = None
        self.previewpics = None
        self.author = self.distribution.get_author()
        self.author_email = self.distribution.get_author_email()
        self.author_website = self.distribution.get_url()
        self.version = self.distribution.get_version()
        self.osversion = None
        self.categories = None
        self.associations = None
        self.clockspeed = None
        self.mkdir = None


    def finalize_options(self):

        #No inadvertent overwrites.
        if os.path.exists(self.outfile) and not self.force:
            raise DistutilsFileError('%s already exists.  Use --force to overwrite.'%self.outfile)

        if self.id is None:
            self.id = self.title.lower().replace(' ','-')

        if self.appdata is None:
            self.appdata = self.id

        #Checks for invalid filename characters.  This list is from Wikipedia.
        #I don't know exactly which characters will make the Pandora choke.
        for i in ('/', '\\', '?', '%', '*', ':', '|', '"', '<', '>'):
            if i in self.id or i in self.appdata:
                raise DistutilsOptionError("You can't use a %s in the id or appdata."%i)

        if self.title == 'UNKNOWN':
            raise DistutilsOptionError('No name was found in the setup script and no title was specified.  You need one of these.')

        if self.description == 'UNKNOWN':
            self.warn('No description was found in the setup script or specified.  You should have one of these.')

        if self.exec_command is None:
            try: self.exec_command = self.distribution.scripts[0]
            except TypeError:
                raise DistutilsOptionError('Either your setup function should specify at least one script (in list format), or you should specify it as an option.  What is your PND going to run?')

        if self.exec_nox and self.exec_xreq:
            raise DistutilsOptionError('You can require X or you can require no X, but how am I supposed to satisfy both?  Please pick only one.')

        if self.icon is not None:
            #Warns if icon doesn't exist.
            #Skip this warning, since it only means anything once packaged, not before.
            #if not os.path.exists(self.icon):
            #   self.warn("Icon file %s doesn't seem to exist."%self.icon)
            #Warns if icon does not appear to be a PNG file (the only valid type?).
            if guess_type(self.icon)[0] != 'image/png':
                self.warn('Only icons in PNG format will work.  Specified icon does not appear to be a PNG.')

        if self.info is not None:
            #Warns if info file doesn't exist.
            #Skip this warning, since it only means anything once packaged, not before.
            #if not os.path.exists(self.info):
            #   self.warn("Info file %s doesn't seem to exist."%self.info)
            #Checks that the info file is an appropriate type.
            self.info_type = guess_type(self.info)[0]
            if self.info_type not in ('text/plain', 'text/html'):
                self.warn("Don't recognize info file extension.  Will assume text/plain")
                self.info_type = 'text/plain'

        if self.previewpics is not None:
            self.previewpics = self.previewpics.split(',')
            #Warns if previewpics don't exist.
            for i in self.previewpics:
                if not os.path.exists(i):
                    self.warn('Preview picture %s does not seem to exist.'%i)

        if self.version == '' or self.version == '0.0.0':
            #Three zeros happens when version is not in setup script.
            self.warn('No version number is specified.  0.0.0.0 will be used.')
        self.version = self.version.split('.')
        if len(self.version) > 4:
            self.warn('Version number has too many dot-separated segments.  Only first four will be used.')
        self.version.extend(('0','0','0','0')) #Ensures at least four subterms.

        if self.osversion is not None:
            self.osversion = self.osversion.split('.')
            if len(self.osversion) > 4:
                self.warn('OS version number has too many dot-separated segments.  Only first four will be used.')
            self.osversion.extend(('0','0','0','0')) #Ensures at least four subterms.

        if self.categories is not None:
            clist = self.categories.split(';')
            self.categories = {}
            for i in clist:
                i = i.split(':')
                if i[0] not in registered_categories:
                    self.warn('%s is not a valid top-level FreeDesktop category.  It will still be added to your PND, but you should only use categories found at http://standards.freedesktop.org/menu-spec/latest/apa.html' % i[0])
                self.categories[i[0]] = []
                if len(i) == 2:
                    subcats = i[1].split(',')
                    if len(subcats) > 1:
                        self.warn('Multiple subcategories under %s may not work properly, but I\'m not entirely sure.'%i[0])
                    for j in subcats:
                        if (i[0] in registered_categories) and (j not in registered_categories[i[0]]):
                            self.warn('%s is not a valid subcategory of %s.  It will still be added to your PND, but you should only use subcategories found at http://standards.freedesktop.org/menu-spec/latest/apa.html' % (j,i[0]))
                        self.categories[i[0]].append(j)
                elif len(i) == 1:
                    self.warn("It's a good idea to specify at least one subcategory under %s"%i[0])
                else:
                    raise DistutilsOptionError('The categories string is improperly formed.')
        else: self.warn('No categories specified.  You should really include at least one.')

        #if self.associations is not None:
            #Do something with associations.

        if self.clockspeed is not None and not self.clockspeed.isdigit():
            self.warn('Your Pandora might not like a non-integer clockspeed.')

        if self.mkdir is not None:
            self.warn("Don't use mkdir.  Please.")
            self.mkdir = self.mkdir.split(',')


    def run(self):
        doc = Document()
        pxml = doc.createElement('PXML')
        doc.appendChild(pxml)
        pxml.setAttribute('xmlns','http://openpandora.org/namespaces/PXML')

        app = doc.createElement('application')
        pxml.appendChild(app)
        app.setAttribute('id', self.id)
        app.setAttribute('appdata', self.appdata)

        title = doc.createElement('title')
        app.appendChild(title)
        title.setAttribute('lang', 'en_US')
        title.appendChild(doc.createTextNode(self.title))

        description = doc.createElement('description')
        app.appendChild(description)
        description.setAttribute('lang', 'en_US')
        description.appendChild(doc.createTextNode(self.description))

        ex = doc.createElement('exec')
        app.appendChild(ex)
        ex.setAttribute('command', self.exec_command)
        if self.exec_args is not None:
            ex.setAttribute('arguments', self.exec_args)
        if self.exec_startdir is not None:
            ex.setAttribute('startdir', self.exec_startdir)
        if self.exec_nostandalone:
            ex.setAttribute('standalone', 'false')
        else:
            ex.setAttribute('standalone', 'true')
        if self.exec_nobackground:
            ex.setAttribute('background', 'false')
        else:
            ex.setAttribute('background', 'true')
        if self.exec_nox:
            ex.setAttribute('x11', 'stop')
        elif self.exec_xreq:
            ex.setAttribute('x11', 'req')

        if self.icon is not None:
            icon = doc.createElement('icon')
            app.appendChild(icon)
            icon.setAttribute('src', self.icon)

        if self.info is not None:
            info = doc.createElement('info')
            app.appendChild(info)
            info.setAttribute('name', '%s help'%self.title)
            info.setAttribute('src', self.info)
            info.setAttribute('type', self.info_type)

        if self.previewpics is not None:
            ppics = doc.createElement('previewpics')
            app.appendChild(ppics)
            for i in self.previewpics:
                pic = doc.createElement('pic')
                ppics.appendChild(pic)
                pic.setAttribute('src', i)

        author = doc.createElement('author')
        app.appendChild(author)
        if self.author != 'UNKNOWN':
            author.setAttribute('name', self.author)
        if self.author_website != 'UNKNOWN':
            author.setAttribute('website', self.author_website)
        if self.author_email != 'UNKNOWN':
            author.setAttribute('email', self.author_email)

        version = doc.createElement('version')
        app.appendChild(version)
        version.setAttribute('major', self.version[0])
        version.setAttribute('minor', self.version[1])
        version.setAttribute('release', self.version[2])
        version.setAttribute('build', self.version[3])

        if self.osversion is not None:
            osversion = doc.createElement('osversion')
            app.appendChild(osversion)
            osversion.setAttribute('major', self.osversion[0])
            osversion.setAttribute('minor', self.osversion[1])
            osversion.setAttribute('release', self.osversion[2])
            osversion.setAttribute('build', self.osversion[3])

        if self.categories is not None:
            categories = doc.createElement('categories')
            app.appendChild(categories)
            for i in self.categories:
                category = doc.createElement('category')
                categories.appendChild(category)
                category.setAttribute('name', i)
                for j in self.categories[i]:
                    subcat = doc.createElement('subcategory')
                    category.appendChild(subcat)
                    subcat.setAttribute('name',j)

        #if self.associations is not None:
            #What do about associations??

        if self.clockspeed is not None:
            clockspeed = doc.createElement('clockspeed')
            app.appendChild(clockspeed)
            clockspeed.setAttribute('frequency', self.clockspeed)

        if self.mkdir is not None:
            mkdir = doc.createElement('mkdir')
            app.appendChild(mkdir)
            for i in self.mkdir:
                dirr = doc.createElement('dir')
                mkdir.appendChild(dirr)
                dirr.setAttribute('path', i)


        #Now that XML is all generated, write it to the specified file.
        outfile = open(self.outfile, 'w')
        #NOTE: If PXML is pretty printed and the PND does not include an icon,
        #then the PND seems to end up unmountable. Therefore, no pretty print.
        #try: outfile.write(doc.toprettyxml())
        try: doc.writexml(outfile)
        finally: outfile.close()
