# -*- coding: utf-8 -*-
from django.db import models
from django.utils.timezone import now


##-----------------------------------------------------------------------------
## Mixins
##-----------------------------------------------------------------------------
class SingletonModelMixin(object):
    """
    Singleton mixin for models.
    """
    __PK = 1

    @classmethod
    def instance(cls):
        """
        Get the singleton instance.

        @return: (object).
        @raise DoesNotExist: if instance does not exist.
        """
        return cls.objects.get(pk=cls.__PK)

    def save(self, *args, **kwargs):
        self.pk = self.__PK
        super(SingletonModelMixin, self).save(*args, **kwargs)


class CreatedOnFieldMixin(object):
    """
    Mixin to specify models that have a `created_on` field
    that is set when creating the instance.
    """
    def save(self, *args, **kwargs):
        if not self.pk:
            self.created_on = now()

        super(CreatedOnFieldMixin, self).save(*args, **kwargs)


##-----------------------------------------------------------------------------
## Managers
##-----------------------------------------------------------------------------
class FilterByFieldManager(models.Manager):
    """
    Manager that retrieves entries based on a field's value.
    """
    def __init__(self, field_name, field_value):
        """
        @param field_name: (str).
        @param field_value: (mixed).
        """
        super(FilterByFieldManager, self).__init__()

        self.__field_name = field_name
        self.__field_value = field_value

    def get_query_set(self):
        return super(FilterByFieldManager, self).get_query_set().filter(**{
            self.__field_name: self.__field_value
        })


class FilterByNullFieldManager(models.Manager):
    """
    Manager that retrieves entries
    based on whether a nullable field is null or not.
    """
    def __init__(self, field_name, is_null):
        """
        @param field_name: (str).
        @param is_null: (bool).
        """
        super(FilterByNullFieldManager, self).__init__()

        self.__field_name = field_name
        self.__is_null = is_null

    def get_query_set(self):
        return super(FilterByNullFieldManager, self).get_query_set().filter(**{
            '%s__isnull' % self.__field_name: self.__is_null,
        })
