# -*- coding: utf-8 -*-
import json
import logging
import traceback

from django.conf import settings
from django.http import Http404, HttpResponse, HttpResponseServerError
from django.utils.decorators import method_decorator
from django.views.decorators.cache import cache_page
from django.views.generic import TemplateView, View


##-----------------------------------------------------------------------------
## Mixins
##-----------------------------------------------------------------------------
class JSONResponseMixin(object):
    """
    Returns data in json format.
    """
    def render_to_response(self, context):
        # return a json response containing 'context' as payload
        return HttpResponse(json.dumps(context),
                            content_type='application/json')


##-----------------------------------------------------------------------------
## Views
##-----------------------------------------------------------------------------
class AjaxView(JSONResponseMixin, View):
    """
    Generic view to handle ajax requests.
    """
    force_ajax = not settings.DEBUG  # force request to be ajax
    error_logger = logging.getLogger('ajax')  # log errors to this logger

    def dispatch(self, request, *args, **kwargs):
        if self.force_ajax and not request.is_ajax():
            raise Http404()

        try:
            return super(AjaxView, self).dispatch(request, *args, **kwargs)
        except Exception:
            if self.error_logger:
                self.error_logger.error(
                    'Ajax call to "%s" returned error: %s.' %
                    (request.build_absolute_uri(), traceback.format_exc()))
            return HttpResponseServerError()

    def get(self, request, *args, **kwargs):
        return self.render_to_response(
            self.handle_get(request, *args, **kwargs))

    def handle_get(self, request, *args, **kwargs):
        """
        Dummy method for handling get requests.
        Must be overridden by implementing class.

        @return: dictionary.
        """
        raise Http404()

    def post(self, request, *args, **kwargs):
        return self.render_to_response(
            self.handle_post(request, *args, **kwargs))

    def handle_post(self, request, *args, **kwargs):
        """
        Dummy method for handling post requests.
        Must be overridden by implementing class.

        @return: dictionary.
        """
        raise Http404()


class RobotsTxtView(TemplateView):
    """
    Shows robots.txt file for search engines.
    Caches pages to maximise page load time & minimise system load.
    """
    template_name = 'robots.txt'

    @method_decorator(cache_page(2592000))  # cache for 30 days
    def dispatch(self, request, *args, **kwargs):
        return super(RobotsTxtView, self).dispatch(request, *args, **kwargs)

    def render_to_response(self, context, **response_kwargs):
        # return a plain text response
        response_kwargs['content_type'] = 'text/plain'

        return super(RobotsTxtView, self)\
            .render_to_response(context, **response_kwargs)
