import datetime
import dateutil.parser
import feedparser

from django.db import models
from django.utils.encoding import smart_unicode as smart_text
from django.utils.timezone import now
from django.utils.translation import ugettext_lazy as _

ua = 'django-agregador/dev +http://bitbucket.org/semente/django-agregador' # TODO
feedparser.USER_AGENT = ua


class Feed(models.Model):
    title = models.CharField(_('Title'), max_length=500)
    feed_url = models.URLField(_('Feed URL'), max_length=500)
    public_url = models.URLField(_('Public URL'), max_length=500)
    is_defunct = models.BooleanField(_('Is defunct?'))
    date_created = models.DateTimeField(_('Date created'), auto_now_add=True)
    date_modified = models.DateTimeField(_('Date modified'), auto_now=True)

    class Meta:
        ordering = ('title',)

    def __unicode__(self):
        return u'%s' % self.title

    @models.permalink
    def get_absolute_url(self):
        return ('feed-detail', None, {
            'pk' : str(self.pk),
        })

    def update(self):
        parsed = feedparser.parse(self.feed_url)
        parsed.entries.reverse()
        for entry in parsed.entries:
            title = smart_text(entry.title, encoding=parsed.encoding)
            guid = smart_text(entry.get('id', entry.link), encoding=parsed.encoding)
            link = smart_text(entry.link, encoding=parsed.encoding)

            if not guid:
                guid = link

            if 'content' in entry:
                content = entry.content[0].value
            elif 'description' in entry:
                content = entry.description
            elif 'summary' in entry:
                content = entry.summary
            else:
                content = u''

            content = smart_text(content, encoding=parsed.encoding)

            if 'published' in entry and entry.published_parsed is not None:
                date = dateutil.parser.parse(entry.published)
            elif 'updated' in entry and entry.updated_parsed is not None:
                date = dateutil.parser.parse(entry.updated)
            else:
                date = now()

            try:
                self.entries.get(guid=guid)
            except Entry.DoesNotExist:
                self.entries.create(title=title, link=link, summary=content,
                                    guid=guid, date=date)


class Entry(models.Model):
    feed = models.ForeignKey(
        Feed,
        verbose_name=_('Feed'),
        related_name='entries'
    )
    title = models.CharField(_('Title'), max_length=500)
    link = models.URLField(_('Link'), max_length=500)
    summary = models.TextField(_('Summary'), blank=True)
    date = models.DateTimeField(_('Date'))
    guid = models.CharField(
        _('GUID'),
        max_length=500,
        db_index=True,
    )
    date_created = models.DateTimeField(_('Date created'), auto_now_add=True)
    date_modified = models.DateTimeField(_('Date modified'), auto_now=True)

    class Meta:
        ordering = ('-date',)
        verbose_name_plural = _('Entries')

    def __unicode__(self):
        return u'%s' % self.title

    @models.permalink
    def get_absolute_url(self):
        return ('entry-detail', None, {
            'feed_pk' : str(self.feed.pk),
            'pk' : str(self.pk),
        })
