# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from django import forms
from django.utils.translation import ugettext_lazy as _
from django.contrib.contenttypes.generic import BaseGenericInlineFormSet

from attrs import settings as ats
from attrs.models import Attr


class AttrFormSet(BaseGenericInlineFormSet):
    _options = {}

    def __init__(self, *args, **kwargs):
        super(AttrFormSet, self).__init__(*args, **kwargs)

        # Check if get_attrs_options method is defined on the model
        # instance, fallback to _attrs_options property then default.
        get_attrs_options = getattr(self.instance, 'get_attrs_options', None)
        if callable(get_attrs_options):
            self._options = get_attrs_options()
        elif hasattr(self.instance, '_attrs_options'):
            self._options = getattr(self.instance, '_attrs_options', {})
        else:
            self._options = ats.DEFAULT_OPTIONS

    def _construct_form(self, i, **kwargs):
        kwargs['options'] = self._options
        return super(AttrFormSet, self)._construct_form(i, **kwargs)

    def clean(self):
        # Validate only if each form is valid.
        if any(self.errors):
            return

        errors = []
        keys = []

        for form in self.forms:
            if 'key' in form.cleaned_data:
                keys.append(form.cleaned_data['key'])

        # Check if max_num is exceeded.
        for key in self._options:
            option = self._options[key]
            max_num = option.get('max_num', 1)

            if max_num and keys.count(key) > max_num:
                errors.append(
                    _('Maximum number for keyword "{}" of "{}" is '
                      'exceeded.'.format(option['name'], max_num)))

        if errors:
            raise forms.ValidationError(errors)


class AttrForm(forms.ModelForm):
    _options = {}
    options = forms.CharField(widget=forms.HiddenInput(), required=False)

    class Meta:
        model = Attr

    def __init__(self, *args, **kwargs):
        # Set options passed in from AttrFormSet's _construct_form.
        self._options = kwargs.pop('options', {})
        super(AttrForm, self).__init__(*args, **kwargs)
        self.fields['options'].initial = self._options

        self.fields['key'].widget = forms.Select(
            choices=self.get_key_choices(),
            attrs={'onchange': 'attrsKeyOnChange(event);'})

        value_choices = self.get_value_choices()
        value_widget = self.get_value_widget()

        if value_choices:
            self.fields['value'].widget = forms.Select(choices=value_choices)
        elif value_widget == ats.WIDGET_TEXTAREA:
            self.fields['value'].widget = forms.Textarea()

    def clean_value(self):
        key = self.cleaned_data.get('key')
        value = self.cleaned_data.get('value')

        try:
            choices = self._options[key]['choices']
            if value not in [x[0] for x in choices]:
                raise forms.ValidationError(
                    _('Value not available for keyword.'))
        except KeyError:
            pass

        return value

    def clean(self):
        # Remove options from cleaned_data.
        if 'options' in self.cleaned_data:
            del self.cleaned_data['options']
        return super(AttrForm, self).clean()

    def has_changed(self):
        # Remove options from changed_data.
        if 'options' in self.changed_data:
            self.changed_data.remove('options')
        return super(AttrForm, self).has_changed()

    def get_key_choices(self):
        choices = (('', '---------'), )
        for key in self._options:
            choices += (key, self._options[key]['name']),
        return choices

    def get_value_choices(self):
        key = self.initial.get('key')
        try:
            return self._options[key]['choices']
        except KeyError:
            return None

    def get_value_widget(self):
        key = self.initial.get('key')
        try:
            return self._options[key]['widget']
        except KeyError:
            return None
