import re

from django import forms
from django.core.exceptions import ValidationError
from django.forms import ModelForm, ModelMultipleChoiceField, \
    MultipleChoiceField, CheckboxSelectMultiple


from avsubmit import validators
from avsubmit.utils import _is_valid_email
from avsubmit.models import SubmissionTarget, SubmissionLog

email_separator_re = re.compile(r',\s*')

class EmailsListField(forms.CharField):
    """
    Field which accepts list of email addresses separated by commas (and
    optionally space(s)).

    """
    def clean(self, value):
        super(EmailsListField, self).clean(value)

        # ok for the field to be empty
        if value == "":
            return

        emails = email_separator_re.split(value)

        for email in emails:
            if not _is_valid_email(email):
                raise ValidationError('%s is not a valid email address.' % email)

        return emails

class UploadForm(forms.Form):
    """
    File upload form object for malware sample submissions.

    """
    sample = forms.FileField(validators=[validators.validate_filesize_limit],
                 widget=forms.FileInput(attrs={'size':'40'}))

class SubmissionLogForm(ModelForm):
    """
    Form to select targets and prepare file submission log entry.

    """
    active_targets = SubmissionTarget.objects.filter(enabled=True)
    submitted_to = ModelMultipleChoiceField(label="Recipients",
                       queryset=active_targets, widget=CheckboxSelectMultiple())
    cc = EmailsListField(required=False, widget=forms.TextInput(
         {
             'style':'width:600px;',
             'placeholder':'(Optional) comma-separated list of addresses',
         }))
    notes = forms.CharField(widget=forms.Textarea(
         {
             'class':'notes',
             'style':'width:600px;',
             'placeholder':'(Required) brief description of file, origins, etc.',
         }))
    class Meta:
        model = SubmissionLog
        fields = ('submitted_to',)

class SubmissionLogSearchForm(forms.Form):
    """
    Form to handle searches for attributes in submission logs.

    """
    search = forms.CharField(widget=forms.TextInput(
         { 'class':'search-query', }),
         help_text='Search by MD5/SHA1/SHA256 hash, file name or submitter')

