"Function to handle packaging and delivery of submitted samples."

import os.path
import logging
import subprocess
from smtplib import SMTPRecipientsRefused

from django.core import mail
from django.conf import settings
from django.contrib import messages
from django.core.exceptions import ObjectDoesNotExist

from avsubmit.models import SubmissionTarget


logger = logging.getLogger('project.default')

def email_sample(filepath, subject, body, from_addr, to_addrs, cc_addrs=None, submitter=None):
    """
    Dispatch email with sample attached to specified recipients.

    - select zip command based on configuration setting; we don't
      use python's zip because it doesn't support encrypting archives
      (see http://docs.python.org/library/zipfile.html)
    - create zip archive of sample, encrypted with password
    - create EmailMessage object with message parameters and
      zip file attachment

    """
    # Shouldn't get recipients passed in that aren't enabled, but validate
    # anyway to handle them.
    for target in to_addrs:
        try:
            if not SubmissionTarget.objects.get(email_address=target).enabled:
                to_addrs.remove(target)
                logger.error("client passed recipient address '%s' but target is not enabled")
        except ObjectDoesNotExist:
            logger.error("client passed recipient address '%s' but target is nonexistent")
            continue
    # Abort if empty recipient list
    if not to_addrs:
        logger.error("client passed empty recipient list (this shouldn't happen)")
        raise Exception("Application error: email submission requested but no valid recipients specified.")

    # Grab password for ZIP archive. If not set, use default 'infected'.
    password = settings.AVSUBMIT_SETTINGS.get('zip_password', 'infected')

    # Grab which binary specified for use to zip samples. Set the 'zip' command
    # as the default. Build archive file name from uncompressed file name.
    zip_cmd = settings.AVSUBMIT_SETTINGS.get('zip_command', 'zip')
    archive_filepath = '%s.zip' % os.path.splitext(filepath)[0]


    # Proceed with handling of the submitted sample
    try:
        if zip_cmd == '7zip':
            subprocess.call(['7z', 'a', '-p%s' % password, '-tzip',
                             archive_filepath, filepath], stdout=open('/dev/null', 'w'))
        elif zip_cmd == 'zip':
            subprocess.call(['zip', '-j', '-P', password, archive_filepath, 
                             filepath], stdout=open('/dev/null', 'w'))
        else:
            # Set incorrectly, abort and send back home
            logger.info("incorrect value specified for zip_cmd: '%s'" % zip_cmd)
            errstr = "Configuration error: invalid ZIP command specified. Configure AVSUBMIT_SETTINGS in your project settings file."
            raise Exception(errstr)
    except OSError as e:
        errstr = "There was a problem packaging sample to ZIP archive for submission. Error received calling archiver: '%s'" % e
        raise Exception(errstr)
    else:
        # No issues zipping sample, move on to sending email(s)
        try:
            # Open single persistent connection to SMTP server that can be reused
            # for all deliveries.
            conn = mail.get_connection()
            conn.open()
        except Exception as e:
            errstr = "Failed to open connection to SMTP host (%s:%d). Sample not sent. Reason: %s." % (conn.host, conn.port, e)
            raise Exception(errstr)

        # Add Cc addresses to list of recipients, if present.
        if cc_addrs is not None:
            to_addrs.extend(cc_addrs)

        failed_addrs       = []
        failed_addr_errors = []
        failed_deliveries  = []
        success_deliveries = []
        for recipient in to_addrs:
            # 'to' must be list or tuple
            to = [recipient]
            # Build email message object & tag submitter's username into
            # X-Submitted-By header.
            message = mail.EmailMessage(subject=subject, body=body, from_email=from_addr,
                                        to=to, connection=conn,
                                        headers={'X-Submitted-By': submitter.username})
            message.attach_file(archive_filepath)
            try:
                message.send()
            except SMTPRecipientsRefused as e:
                failed_addrs.append(recipient)
                failure = "Error %d (%s)" % (e.recipients[recipient][0], e.recipients[recipient][1])
                failed_addr_errors.append(failure)
            else:
                if cc_addrs is not None and recipient in cc_addrs:
                    continue  # Don't add Cc addresses to successful delivery
                              # list since those can't be logged with relations.
                success_deliveries.append(recipient)
        conn.close()

        # Prepare data to return for tracking results attempting delivery to
        # provided addresses. All failures and no delivery successes results in
        # an exception and error. Some failures with at least one successful
        # delivery displays confirmation and error messages but continues and
        # logs successful dispatches accordingly.
        if len(failed_addrs) > 0:
            failed_deliveries = zip(failed_addrs, failed_addr_errors)
            suf = "" if len(failed_addrs) == 1 else "es"
            errstr = "Delivery to address%s failed: %s" % (suf, 
                      '; '.join(["%s: %s" % (fail[0], fail[1]) for fail in failed_deliveries]))
            if len(success_deliveries) == 0:
                raise Exception(errstr)

        return (success_deliveries, failed_deliveries)
    finally:
        try:
            for f in (filepath,archive_filepath):
                os.remove(f)
        except OSError as e:
            # for whatever reason the file wasn't there/removable
            pass

