import os
import logging
import subprocess
from hashlib import md5, sha1, sha256

from django.db.models import Q
from django.conf import settings
from django.contrib import messages
from django.http import HttpResponse
from django.template import RequestContext
from django.contrib.auth.decorators import login_required
from django.shortcuts import render, render_to_response, redirect
from django.core.paginator import Paginator, InvalidPage, EmptyPage
from django.http import Http404, HttpResponseBadRequest, HttpResponseRedirect

from avsubmit.forms import *
from avsubmit.utils import _is_valid_email
from avsubmit.samplehandler import email_sample
from avsubmit.models import SubmissionTarget, SubmissionLog


logger = logging.getLogger('project.default')

@login_required
def upload_file(request):
    """
    Presents form to handle file uploads for sample submissions.

    """
    # User must have file submit permission
    if not request.user.has_perm('avsubmit.submit_file'):
        logger.error('access denied submitting sample for user %s' % request.user.username)
        messages.error(request, 'Access denied: you do not have permission to submit samples.')
        return redirect('avsubmit_index')

    # Make sure at least one active submission target to send to
    if SubmissionTarget.objects.filter(enabled=True).count() == 0:
        messages.warning(request, 'You must create and enable a submission target before uploading files.')
        return redirect('avsubmit_index')

    # Validate upload related settings. Error on missing or invalid setting
    # values.
    if not settings.AVSUBMIT_SETTINGS.get('sender_address'):
        messages.error(request, "Configuration error: submission sender address not found. Configure AVSUBMIT_SETTINGS in your project settings file.")
        return redirect('avsubmit_index')
    elif not _is_valid_email(settings.AVSUBMIT_SETTINGS.get('sender_address')):
        messages.error(request, "Configuration error: submission sender address ('%s') is invalid. Configure AVSUBMIT_SETTINGS properly in your project settings file." % settings.AVSUBMIT_SETTINGS['sender_address'])
        return redirect('avsubmit_index')
    elif not settings.AVSUBMIT_SETTINGS.get('max_upload_size'):
        messages.error(request, "Configuration error: maximum upload size not set. Configure AVSUBMIT_SETTINGS in your project settings file.")
        return redirect('avsubmit_index')
    elif not str(settings.AVSUBMIT_SETTINGS['max_upload_size']).isdigit():
        messages.error(request, "Configuration error: maximum upload size setting ('%s') is invalid. Configure AVSUBMIT_SETTINGS properly in your project settings file." % settings.AVSUBMIT_SETTINGS['max_upload_size'])
        return redirect('avsubmit_index')

    if request.method == 'POST':
        form = UploadForm(request.POST, request.FILES)
        if form.is_valid():
            sample = request.FILES['sample']
            sample.magic = subprocess.Popen(['file', '-b', sample.temporary_file_path()],
                                            stdout=subprocess.PIPE).communicate()[0].strip()

            # Calculate file hashes
            md5_sum    = md5()
            sha1_sum   = sha1()
            sha256_sum = sha256()
            while True:
                filedata = sample.read(10240)
                if len(filedata) == 0:
                    break
                md5_sum.update(filedata)
                sha1_sum.update(filedata)
                sha256_sum.update(filedata)
            sample.md5    = md5_sum.hexdigest()
            sample.sha1   = sha1_sum.hexdigest()
            sample.sha256 = sha256_sum.hexdigest()

            # Save uploaded file to home on disk to work with in submission stage.
            dest = open('/tmp/%s.bin' % sample.md5, 'wb')
            for chunk in sample.chunks():
                dest.write(chunk)
            sample.filepath = dest.name

            # Resultant file attributes:
            #  sample.name
            #  sample.size
            #  sample.magic
            #  sample.md5
            #  sample.sha1
            #  sample.sha256
            #  sample.filepath

            # Store uploaded sample data in session for submission view.
            request.session['avsubmit_upload'] = {
                'uploaded': True,
                'name':     sample.name,
                'size':     sample.size,
                'magic':    sample.magic,
                'md5':      sample.md5,
                'sha1':     sample.sha1,
                'sha256':   sample.sha256,
                'filepath': sample.filepath,
            }
            request.session.modified = True

            return redirect('submit_file')
    else:
        form = UploadForm()

    return render_to_response('upload_file.html', {
        'form': form,
    }, context_instance=RequestContext(request))


@login_required
def submit_file(request):
    """
    Presents secondary form to display file details and email sample to selected
    recipients.

    """
    # Shouldn't be at this view unless a file has been uploaded for submission.
    if ('avsubmit_upload' not in request.session or
        request.session['avsubmit_upload']['uploaded'] != True):

        return redirect('upload_file')

    if request.method == 'POST':
        form = SubmissionLogForm(request.POST)
        if form.is_valid():
            to = form.cleaned_data['submitted_to']
            cc = form.cleaned_data.get('cc', '')    # empty string needed
            notes = form.cleaned_data['notes']
            subject = "Sample submission %s" % request.session['avsubmit_upload']['md5']
            submitter = request.user
            filepath = request.session['avsubmit_upload']['filepath']
            password = settings.AVSUBMIT_SETTINGS.get('zip_password', 'infected')

            # Build message body for email submissions
            msgtext = 'Password: {password}\n\nSubmitter notes:\n\n{notes}\n'.format(
                       password=password, notes=notes)

            # Attempt to submit sample to specified recipients. email_sample()
            # tries each specified address in turn. Complete delivery failures
            # result in no log and return user to home. Partial failures display
            # an error for failed targets but continue to submit and log
            # submission appropriately.
            from_addr = settings.AVSUBMIT_SETTINGS['sender_address']
            try:
                (success_deliveries, failed_deliveries) = email_sample(
                    filepath, subject, msgtext, from_addr,
                    [x.email_address for x in to], cc, submitter)

                # Queue up submission log based on initial data from file upload
                # and submission form.
                log = form.save(commit=False)
                log.file_name = request.session['avsubmit_upload']['name']
                log.file_size = request.session['avsubmit_upload']['size']
                log.file_md5 = request.session['avsubmit_upload']['md5']
                log.file_sha1 = request.session['avsubmit_upload']['sha1']
                log.file_sha256 = request.session['avsubmit_upload']['sha256']
                log.file_magic = request.session['avsubmit_upload']['magic']
                log.date_submitted = None
                log.notes = notes
                log.submitter = submitter
            except Exception as e:
                logger.error(e)
                messages.error(request, e)
                return redirect('avsubmit_index')
            finally:
                # Clear data from session.
                del request.session['avsubmit_upload']

            # Deal with delivery successes
            success_text = ', '.join(success_deliveries)
            for recipient in success_deliveries:
                logger.info("%s submitted %s/%s to %s" % (submitter, log.file_name,
                            log.file_md5, recipient))
            suf = "" if len(success_deliveries) == 1 else "s"
            messages.success(request, 'Sample submitted to %d recipient%s (%s)' % (
                             len(success_deliveries), suf, success_text))

            # Deal with delivery failures
            if len(failed_deliveries) > 0:
                suf = "" if len(failed_deliveries) == 1 else "es"
                errstr = "Delivery to %d address%s failed: %s" % (len(failed_deliveries), suf,
                           '; '.join(["%s: %s" % (fail[0], fail[1]) for fail in failed_deliveries]))
                # Fail return object should be a list of tuples as (address, reason).
                for error in failed_deliveries:
                    logger.info("%s submission failed %s/%s to %s: %s" % (submitter, log.file_name,
                            log.file_md5, error[0], error[1]))
                messages.warning(request, errstr)

            # Outcome of message deliveries is known. Save submission log to DB
            # and set correct recipients list.
            log.save()
            form.save_m2m()
            log.submitted_to = SubmissionTarget.objects.filter(
                email_address__in=success_deliveries)

            return redirect('submissionlog_detail', pk=log.id)
    else:
        form = SubmissionLogForm()

    return render_to_response('submit_file.html', {
        'form': form,
    }, context_instance=RequestContext(request))


@login_required
def list_submissions(request):
    """
    Presents paginated list of submission logs. This view considers that
    other query parameters may have been passed in and adds them into
    the context so the template can include them in pagination URLs.

    """
    # Validate logs per page setting
    logs_per_page = settings.AVSUBMIT_SETTINGS.get('logs_per_page', 15)
    if not str(logs_per_page).isdigit():
        errstr = "Configuration error: logs per page setting ('%s') is invalid. Configure AVSUBMIT_SETTINGS properly in your project settings file." % logs_per_page
        logger.error(errstr)
        messages.error(request, errstr)
        return redirect('avsubmit_index')

    search_form = SubmissionLogSearchForm()
    query = request.GET.get('search', '')
    if query:
        qset = (
            Q(file_name__icontains=query) |
            Q(file_md5=query) |
            Q(file_sha1=query) |
            Q(file_sha256=query) |
            Q(submitter__username=query)
        )
        submissionlog_list = SubmissionLog.objects.filter(qset)

        # If search query returned a single object, skip the paged list display
        # and jump right to the relevant result.
        if submissionlog_list.count() == 1:
            return redirect('submissionlog_detail', pk=submissionlog_list.all()[0].id)

    else:
        # XXX This should be limited to a reasonable size, say 200 instead of all
        submissionlog_list = SubmissionLog.objects.all()

    paginator = Paginator(submissionlog_list, int(logs_per_page))  # show N logs per page
    
    # Make sure page request is an int. If not, deliver first page.
    try:
        page = int(request.GET.get('page', '1'))
    except ValueError:
        page = 1

    # If page request (9999) is out of range, deliver last page of results.
    try:
        submissionlogs = paginator.page(page)
    except (EmptyPage, InvalidPage):
        submissionlogs = paginator.page(paginator.num_pages)

    # Determine other parameters from request, provide to template to craft
    # smarter URLs that handle concurrent search + pagination (etc.).
    if request.method == 'GET':
        more_query_params = ''
        for key, value in request.GET.items():
            if key != 'page':
                more_query_params += '&%s=%s' % (key, value)

    return render_to_response('submissionlog_pages.html', {
        'submissionlogs': submissionlogs,
        'search_form': search_form,
        'more_query_params': more_query_params,
    }, context_instance=RequestContext(request))

@login_required
def index(request):
    """
    Presents application home page.

    """
    upload_form = UploadForm()
    search_form = SubmissionLogSearchForm()
    submissionlog_count = SubmissionLog.objects.count()
    submissionlog_latest = SubmissionLog.objects.all()[:10]
    submissiontarget_count = SubmissionTarget.objects.count()

    return render_to_response('index.html', {
        'upload_form': upload_form,
        'search_form': search_form,
        'submissionlog_count': submissionlog_count,
        'submissionlog_latest': submissionlog_latest,
        'submissiontarget_count': submissiontarget_count,
    }, context_instance=RequestContext(request))

