import time

import boto
import boto.sdb
import boto.exception

from django.contrib.sessions.backends.base import SessionBase, CreateError

from django_aws import utils
from django_aws import conf


_domain = None


class SessionStore(SessionBase):
    def __init__(self, session_key=None):
        global _domain

        if _domain is None:
            if conf.AWS_SESSIONS_SDB_REGION is None:
                connection = boto.connect_sdb()
            else:
                connection = boto.sdb.connect_to_region(
                    conf.AWS_SESSIONS_SDB_REGION)

            _domain = connection.create_domain(conf.AWS_SESSIONS_SDB_DOMAIN)

        self.domain = _domain
        super(SessionStore, self).__init__(session_key)

    def exists(self, session_key):
        return self.domain.get_item(session_key) is not None

    def create(self):
        while True:
            self._session_key = self._get_new_session_key()
            try:
                self.save(must_create=True)
            except CreateError:
                continue

            self.modified = True
            return

    def save(self, must_create=False):
        session_key = self._get_or_create_session_key()

        attributes = {
            'session_data': self.encode(self._get_session(no_load=must_create)),
            'expire_timestamp': utils.dt2ts(self.get_expiry_date())
        }

        try:
            self.domain.put_attributes(
                session_key,
                attributes,
                expected_value=('session_data', False) if must_create else None)
        except boto.exception.SDBResponseError:
            if must_create:
                raise CreateError
            raise

    def delete(self, session_key=None):
        item = self.domain.get_item(session_key)
        if item is not None:
            item.delete()

    def load(self):
        data = self.domain.get_item(self.session_key)

        if data is not None:
            return self.decode(data['session_data'])

        self.create()
        return {}


    @classmethod
    def clear_expired(cls):
        session_store = cls()

        query = 'select * from `{}` where expire_timestamp < {}'.format(
            conf.AWS_SESSIONS_SDB_DOMAIN, utils.utctime())

        for item in session_store.domain.select(query):
            item.delete()
