from django.conf import settings
from django.template.loader import render_to_string
from django.forms import ModelForm
from django.core.mail import EmailMultiAlternatives
from django.contrib.sites.models import RequestSite
from django.contrib.sites.models import Site
from django.template import RequestContext


class BaseContactFormForm(ModelForm):

    def __init__(self, data=None, files=None, request=None, *args, **kwargs):
        if request is None:
            raise TypeError("Keyword argument 'request' must be supplied")
        super(BaseContactFormForm, self).__init__(data=data, files=files, *args, **kwargs)
        self.request = request

    from_email = settings.DEFAULT_FROM_EMAIL
    recipients = [x[1] for x in settings.CONTACT_FORM_RECIPIENTS]
    text_template = 'contact/email_body.txt'
    html_template = 'contact/html_body.html'
    subject_template_name = 'contact/subject.txt'

    def save(self, commit=True, *args, **kwargs):
        instance = super(BaseContactFormForm, self).save(commit=commit)
        context = self.get_context()
        context.update({'url': instance.edit_url})
        text_content = render_to_string(self.text_template, context)
        html_content = render_to_string(self.html_template, context)
        msg = EmailMultiAlternatives(self.get_subject(), text_content, self.from_email, self.recipients)
        msg.attach_alternative(html_content, "text/html")
        msg.send(fail_silently=False)
        self.data = {}
        return instance

    def get_context(self):
        if not self.is_valid():
            raise ValueError("Cannot generate Context from invalid contact form")
        if Site._meta.installed:
            site = Site.objects.get_current()
        else:
            site = RequestSite(self.request)
        return RequestContext(self.request, dict(self.cleaned_data, site=site))

    def get_subject(self):
        subject = render_to_string(self.subject_template_name, self.get_context())
        return ''.join(subject.splitlines())
