import logging
import warnings
from django.http import HttpResponse, HttpResponseRedirect
from django.http import HttpResponsePermanentRedirect, HttpResponseGone
from django.template import loader, RequestContext


warnings.warn(
    'Function-based generic views have been deprecated; use class-based views '
    'instead.', PendingDeprecationWarning
)


logger = logging.getLogger('cbv')


def direct_to_template(request, template, extra_context=None, mimetype=None,
                       **kwargs):
    """
    Render a given template with any extra URL parameters in the context as
    ``{{ params }}``.
    """
    if extra_context is None: extra_context = {}
    dictionary = {'params': kwargs}
    for key, value in extra_context.items():
        if callable(value):
            dictionary[key] = value()
        else:
            dictionary[key] = value
    c = RequestContext(request, dictionary)
    t = loader.get_template(template)
    return HttpResponse(t.render(c), mimetype=mimetype)

def redirect_to(request, url, permanent=True, query_string=False, **kwargs):
    """
    Redirect to a given URL.

    The given url may contain dict-style string formatting, which will be
    interpolated against the params in the URL.  For example, to redirect from
    ``/foo/<id>/`` to ``/bar/<id>/``, you could use the following URLconf::

        urlpatterns = patterns('',
            ('^foo/(?P<id>\d+)/$', 'django.views.generic.simple.redirect_to', {'url' : '/bar/%(id)s/'}),
        )

    If the given url is ``None``, a HttpResponseGone (410) will be issued.

    If the ``permanent`` argument is False, then the response will have a 302
    HTTP status code. Otherwise, the status code will be 301.

    If the ``query_string`` argument is True, then the GET query string
    from the request is appended to the URL.

    """
    args = request.META["QUERY_STRING"]
    if args and query_string and url is not None:
        url = "%s?%s" % (url, args)

    if url is not None:
        if permanent:
            return HttpResponsePermanentRedirect(url % kwargs)
        return HttpResponseRedirect(url % kwargs)
    else:
        logger.warning(
            'Gone: %s' % request.path,
            extra={
                'status_code': 410,
                'request': request
            })
        return HttpResponseGone()
