# Copyright (c) 2014  Niklas Rosenstein
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
r"""
    chang.jinja2 - Django Jinja2 binding
    ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    The :mod:`chang.jinja2` module implements a connection to Django
    for rendering Jinja2 templates. It allows to let Django and Jinja2
    templates to live besides each other without obstructing the other.

    This module is highly influenced by :mod:`django_jinja`, which did
    however not fit my needs completely.

    Django Settings
    ---------------

    * `CHANG_JINJA2_ENV`: A fully constructed :class:`jinja2.Environment`
    object. If this is not given or None, a new Environment will be
    created from other settings.
    * `CHANG_JINJA2_ENVKWARGS`: Keyword arguments to create the
    :class:`jinja2.Environment` from if `CHANG_JINJA2_ENV` is not
    supplied in the settings.
    * `CHANG_JINJA2_EXTENSIONS`: A tuple of extensions that the Jinja2
    Environment should use. Only used when `CHANG_JINJA2_ENV` is not
    given.
    * `CHANG_JINJA2_LOADER`: A Jinja2 Loader object or None.
    * `CHANG_JINJA2_VERIFY`: A callback that will be passed the
    name of the template and should return True when it should be
    loaded as a Jinja2 template.
    * `CHANG_JINJA2_ALWAYS`: If this is True, `CHANG_JINJA2_VERIFY`
    will not be used and all templates will be loaded as Jinja2
    templates.
    * `CHANG_JINJA2_DJANGOGLOBALS`: Default is True. Expose Django filters
    and functions to Jinja2 when this is set to True.
    * `CHANG_JINJA2_INCLUDEDJANGOCONFIG`: Default is True. Expose the
    Django settings to the Jinja2 Environment globals.
"""

from __future__ import absolute_import

from . import utils
from django.conf import settings
from django.template import TemplateDoesNotExist
from django.template.context import BaseContext as _BaseContext
from django.template.loaders.filesystem import Loader as _FileSystemLoader
from django.template.loaders.app_directories import Loader as _AppDirectoriesLoader
from django.template.loaders.app_directories import app_template_dirs
from django.contrib.staticfiles.storage import staticfiles_storage
from django.core import urlresolvers
import six
import jinja2.ext

def filter_static(path):
    return staticfiles_storage.url(path)

def filter_reverse(value, *args, **kwargs):
    return urlresolvers.reverse(value, args=args, kwargs=kwargs)

def dict_from_context(context):
    if isinstance(context, _BaseContext):
        new_dict = {}
        for item in tuple(context):
            new_dict.update(item)
        return new_dict

    return dict(context)

class Template(jinja2.Template):
    r""" Customized Template class to match with the Django
    template rendering. """

    def render(self, context={}):
        context = dict_from_context(context)
        return super(Template, self).render(context)

    def stream(self, context={}):
        context = dict_from_context(context)
        return super(Template, self).stream(context)

class TemplateLoaderMixin(object):

    def load_template(self, template_name, template_dirs=None):
        if not _verify(template_name):
            return super(TemplateLoaderMixin, self).load_template(
                        template_name, template_dirs)

        try:
            template = env.get_template(template_name)
            return template, template.filename
        except jinja2.TemplateNotFound:
            raise TemplateDoesNotExist(template_name)

class FileSystemLoader(TemplateLoaderMixin, _FileSystemLoader):
    pass
class AppDirectoriesLoader(TemplateLoaderMixin, _AppDirectoriesLoader):
    pass

class RegisterHelper(object):
    r""" An instance of this class represents the global chang.jinja2
    environment. It can be used to register global functions, data and
    filters. """

    def __getitem__(self, name):
        return env.globals[name]

    def __setitem__(self, name, value):
        env.globals[name] = value

    def function(self, func):
        env.globals[func.func_name] = func
        return func

    def filter(self, func):
        env.filters[func.func_name] = func
        return func

register = RegisterHelper

class CsrfExtension(jinja2.ext.Extension):
    r""" Adds a {% csrf %} tag to Jinja. """

    tags = set(['csrf', 'csrf_token'])
    template = u'<input type="hidden" name="csrfmiddlewaretoken" value="%s">'

    def parse(self, parser):
        lineno = next(parser.stream).lineno
        ctx_ref = jinja2.nodes.ContextReference()
        node = self.call_method('_render_csrf', [ctx_ref], lineno=lineno)
        return jinja2.nodes.CallBlock(node, [], [], [], lineno=lineno)

    def _render_csrf(self, context, caller):
        csrf_token = context['csrf_token']
        return jinja2.Markup(self.template % unicode(csrf_token))

csrf = CsrfExtension

# Get the settings from tje Django configuration and either
# use the specified Jinja2 Environment or create a new one.
env = getattr(settings, 'CHANG_JINJA2_ENV', None)
if env is None:
    kwargs = getattr(settings, 'CHANG_JINJA2_ENVKWARGS', {})
    extensions = getattr(settings, 'CHANG_JINJA2_EXTENSIONS', None)
    if extensions is None:
        extensions = (
            'jinja2.ext.do',
            'jinja2.ext.loopcontrols',
            'jinja2.ext.with_',
            'jinja2.ext.i18n',
            'jinja2.ext.autoescape',
            'chang.jinja2.csrf',
        )
    if extensions:
        kwargs['extensions'] = extensions
    kwargs.setdefault('autoescape', True)
    env = jinja2.Environment(**kwargs)

# Update the template class for the Environment.
env.template_class = Template

# Update the loader of the environment.
_loader = getattr(settings, 'CHANG_JINJA2_LOADER', None)
if _loader:
    env.loader = _loader
elif not env.loader:
    env.loader = jinja2.FileSystemLoader(
                app_template_dirs + tuple(settings.TEMPLATE_DIRS))

# Retreive from the settings how we will determine that a
# template should be loaded with Jinja.
if getattr(settings, 'CHANG_JINJA2_ALWAYS', False):
    _verify = lambda n: True
else:
    _verify = lambda n: n.endswith('.jinja')
    _verify = getattr(settings, 'CHANG_JINJA2_VERIFY', _verify)

# Update the Environment globals when the Django filters and
# functions should be exposed.
if getattr(settings, 'CHANG_JINJA2_DJANGOGLOBALS', True):
    # From django-jinja/base.py v0.23
    filters = {
        'static': "chang.jinja2.filter_static",
        'reverseurl': "chang.jinja2.filter_reverse",
        'addslashes': "django.template.defaultfilters.addslashes",
        'capfirst': "django.template.defaultfilters.capfirst",
        'escapejs': "django.template.defaultfilters.escapejs_filter",
        'fix_ampersands': "django.template.defaultfilters.fix_ampersands_filter",
        'floatformat': "django.template.defaultfilters.floatformat",
        'iriencode': "django.template.defaultfilters.iriencode",
        'linenumbers': "django.template.defaultfilters.linenumbers",
        'make_list': "django.template.defaultfilters.make_list",
        'slugify': "django.template.defaultfilters.slugify",
        'stringformat': "django.template.defaultfilters.stringformat",
        'truncatechars': "django.template.defaultfilters.truncatechars",
        'truncatewords': "django.template.defaultfilters.truncatewords",
        'truncatewords_html': "django.template.defaultfilters.truncatewords_html",
        'urlizetrunc': "django.template.defaultfilters.urlizetrunc",
        'ljust': "django.template.defaultfilters.ljust",
        'rjust': "django.template.defaultfilters.rjust",
        'cut': "django.template.defaultfilters.cut",
        'linebreaksbr': "django.template.defaultfilters.linebreaksbr",
        'linebreaks': "django.template.defaultfilters.linebreaks_filter",
        'removetags': "django.template.defaultfilters.removetags",
        'striptags': "django.template.defaultfilters.striptags",
        'add': "django.template.defaultfilters.add",
        'date': "django.template.defaultfilters.date",
        'time': "django.template.defaultfilters.time",
        'timesince': "django.template.defaultfilters.timesince_filter",
        'timeuntil': "django.template.defaultfilters.timeuntil_filter",
        'default_if_none': "django.template.defaultfilters.default_if_none",
        'divisibleby': "django.template.defaultfilters.divisibleby",
        'yesno': "django.template.defaultfilters.yesno",
        'pluralize': "django.template.defaultfilters.pluralize",

        'title': "django.template.defaultfilters.title",
        'upper': "django.template.defaultfilters.upper",
        'lower': "django.template.defaultfilters.lower",
        'urlencode': "django.template.defaultfilters.urlencode",
        'urlize': "django.template.defaultfilters.urlize",
        'wordcount': "django.template.defaultfilters.wordcount",
        'wordwrap': "django.template.defaultfilters.wordwrap",
        'center': "django.template.defaultfilters.center",
        'join': "django.template.defaultfilters.join",
        'length': "django.template.defaultfilters.length",
        'random': "django.template.defaultfilters.random",
        'default': "django.template.defaultfilters.default",
        'filesizeformat': "django.template.defaultfilters.filesizeformat",
        'pprint': "django.template.defaultfilters.pprint",
    }
    globals = {
        'url': "chang.jinja2.filter_reverse",
        'static': "chang.jinja2.filter_static",
    }

    for name, value in filters.iteritems():
        if isinstance(value, six.string_types):
            value = utils.load_class(value)
        env.filters[name] = value
    for name, value in globals.iteritems():
        if isinstance(value, six.string_types):
            value = utils.load_class(value)
        env.globals[name] = value

# Make the Django configuration values available to the Jinja2
# templates if that is desired by the settings.
if getattr(settings, 'CHANG_JINJA2_INCLUDEDJANGOCONFIG', True):
    for key in dir(settings):
        if not (key.startswith('_') or key in env.globals):
            env.globals[key] = getattr(settings, key)


