# Copyright (c) 2014  Niklas Rosenstein
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
r"""
    chang.local - Local Proxy Middleware
    ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    This module provides a Django Middleware that makes the current
    request available as a :mod:`werkzeug` local proxy. It can be
    accessed through :data:`chang.local.request`.

"""

import threading
from werkzeug.local import LocalProxy

class LocalMiddleware(object):
    r""" This Django middleware processes a request and puts it
    on a place that is associated with the current thread ID. The
    request can later be accessed via a :class:`LocalProxy` from
    this module. (see :data:`request`). """

    def process_request(self, request):
        if hasattr(_request_local, 'request'):
            raise RuntimeError('thread request already reserved')
        _request_local.request = request

    def process_response(self, request, response):
        if not hasattr(_request_local, 'request'):
            raise RuntimeError('no request for thread reserved')

        del _request_local.request
        return response

Middleware = LocalMiddleware

# This map associates a thread identifier with the request
# object that is currently handled by the thread. It is put
# there by LocalMiddleware.process_request() and removed in
# LocalMiddleware.process_response().
_request_local = threading.local()

# A proxy for the current request that is being processed
# by the current thread. The LocalMiddleware class must be
# installed in the Django settings.
@LocalProxy
def request():
    if not hasattr(_request_local, 'request'):
        raise RuntimeError('working outside of request context')

    return _request_local.request

