# Copyright (c) 2014  Niklas Rosenstein
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
r"""
    chang.utils
    ~~~~~~~~~~~

"""

from importlib import import_module
from django.shortcuts import render
from django.core.exceptions import ImproperlyConfigured

class TemplateRenderer(object):
    r""" This is a helper class to manage rendering templates (not
    specifically Jinja templates but any) and constructing a default
    context for templates rendered with this renderer.

    .. code-block:: python

        @TemplateRenderer
        def render(context):
            context['manager'] = get_global_manager()
            context['time'] = now()
            # ...
            return context

        # ...
        render(request, 'foo.jinja', context={'post': post})
        render.as_response(request, 'foo.jinja', context={'post': post}')

    """

    def __init__(self, default_context_provider=None):
        super(TemplateRenderer, self).__init__()
        self.default_context_provider = default_context_provider
        self.context_class = dict

    def __call__(self, request, template_name, context={}):

        context = self.context_class(context)
        if self.default_context_provider:
            new_context = self.default_context_provider(context)
            if new_context is None:
                new_context = context

        return render(request, template_name, context)

def load_class(name):
    r""" Loads a class by name and returns it. """

    mod_name, class_name = name.rsplit('.', 1)

    try:
        mod = import_module(mod_name)
    except ImportError:
        raise ImproperlyConfigured('No module %r' % mod_name)

    try:
        return getattr(mod, class_name)
    except AttributeError:
        raise ImproperlyConfigured('Module %r does not define a class %r' %
                (mod_name, class_name))

