# -*- coding: utf-8 -*-
import warnings

from django.core.exceptions import ImproperlyConfigured

from cms.app_base import CMSApp
from cms.exceptions import AppAlreadyRegistered
from cms.utils.conf import get_cms_setting
from cms.utils.django_load import load, iterload_objects


class ApphookPool(object):

    def __init__(self):
        self.apphooks = []
        self.apps = {}
        self.discovered = False

    def clear(self):
        # TODO: remove this method, it's Python, we don't need it.
        self.apphooks = []
        self.apps = {}
        self.discovered = False

    def register(self, app, discovering_apps=False):
        if self.apphooks and not discovering_apps:
            return

        if app.__name__ in self.apps:
            raise AppAlreadyRegistered(
                'A CMS application %r is already registered' % app.__name__)

        if not issubclass(app, CMSApp):
            raise ImproperlyConfigured(
                'CMS application must inherit from cms.app_base.CMSApp, '
                'but %r does not' % app.__name__)

        if not hasattr(app, 'menus') and hasattr(app, 'menu'):
            warnings.warn("You define a 'menu' attribute on CMS application %r, "
                "but the 'menus' attribute is empty, did you make a typo?" % app.__name__)

        self.apps[app.__name__] = app

    def discover_apps(self):
        self.apphooks = get_cms_setting('APPHOOKS')

        if self.apphooks:
            for cls in iterload_objects(self.apphooks):
                try:
                    self.register(cls, discovering_apps=True)
                except AppAlreadyRegistered:
                    pass

        else:
            load('cms_app')

        self.discovered = True

    def get_apphooks(self):
        hooks = []

        if not self.discovered:
            self.discover_apps()

        for app_name in self.apps:
            app = self.apps[app_name]

            if app.urls:
                hooks.append((app_name, app.name))

        # Unfortunately, we loose the ordering since we now have a list of tuples. Let's reorder by app_name:
        hooks = sorted(hooks, key=lambda hook: hook[1])

        return hooks

    def get_apphook(self, app_name):
        if not self.discovered:
            self.discover_apps()

        try:
            return self.apps[app_name]
        except KeyError:
            # deprecated: return apphooks registered in db with urlconf name instead of apphook class name
            for app in self.apps.values():
                if app_name in app.urls:
                    return app

        raise ImproperlyConfigured('No registered apphook %r found' % app_name)


apphook_pool = ApphookPool()
