# -*- coding: utf-8 -*-

from django.conf import settings
from django.core.urlresolvers import reverse
from django.contrib.sites.models import get_current_site
from django.core.mail import send_mail, BadHeaderError
from django.core.exceptions import ImproperlyConfigured
from django.http import HttpResponse
from django.template.loader import render_to_string
from django.views.generic import FormView, TemplateView


from contato.forms import ContactForm


class SentView(TemplateView):
    template_name = 'contato/sent.html'


class BaseContactView(FormView):
    body_template = None
    subject_template = None
    recipient_list = None

    def get_success_url(self):
        return reverse('contato-sent')

    def get_body_template(self):
        if self.body_template:
            template = self.body_template
        else:
            raise ImproperlyConfigured('Please provide a body_template')
        return template

    def get_subject_template(self):
        if self.subject_template:
            template = self.subject_template
        else:
            raise ImproperlyConfigured('Please provide a subject_template')
        return template

    def get_recipient_list(self):
        if self.recipient_list:
            recipient_list = self.recipient_list
        else:
            try:
                recipient_list = settings.CONTATO_RECIPIENT_LIST
            except AttributeError:
                raise ImproperlyConfigured('Please provide a recipient_list')
        return recipient_list

    def get_sender_email(self, form_data):
        raise NotImplementedError

    def get_subject(self, form_data):
        raise NotImplementedError

    def get_body(self, form_data):
        raise NotImplementedError

    def send(self, form_data, fail_silently=False):
        subject, body = self.get_subject(form_data), self.get_body(form_data)
        sender_email = self.get_sender_email(form_data)
        recipient_list = self.get_recipient_list()
        send_mail(subject, body, sender_email, recipient_list, fail_silently=fail_silently)

    def form_valid(self, form):
        try:
            self.send(form.cleaned_data)
        except BadHeaderError:
            return HttpResponse('Invalid header found.')
        return super(BaseContactView, self).form_valid(form)


class ContactView(BaseContactView):
    form_class = ContactForm
    template_name = 'contato/contato_form.html'
    body_template = 'contato/body.txt'
    subject_template = 'contato/subject.txt'
    site_name = None

    def get_site_name(self):
        if self.site_name:
            site_name = self.site_name
        else:
            site_name = get_current_site(self.request).name
        return site_name

    def get_sender_email(self, form_data):
        name, email = form_data.get('name'), form_data.get('email')
        return '%s <%s>' % (name, email)

    def get_subject(self, form_data):
        site_name = self.get_site_name()
        template = self.get_subject_template()
        string = render_to_string(template, {'site_name': site_name})
        return string.strip()

    def get_body(self, form_data):
        message = form_data.get('message')
        phone_number = form_data.get('phone_number', None)
        template = self.get_body_template()
        context = dict(message=message, phone_number=phone_number)
        string = render_to_string(template, context)
        return string.lstrip()
