# -*- coding: utf-8 -*-

from django.conf import settings
from django.contrib.sites.models import get_current_site
from django.core.exceptions import ImproperlyConfigured
from django.core.mail import BadHeaderError, EmailMessage
from django.core.urlresolvers import reverse
from django.http import HttpResponse
from django.template.loader import render_to_string
from django.utils.translation import ugettext_lazy as _
from django.views.generic import FormView, TemplateView

from contato.forms import AntiSpamContactForm, ContactForm


class SentView(TemplateView):
    template_name = 'contato/sent.html'


class BaseContactView(FormView):
    body_template = None
    subject_template = None
    recipient_list = None

    def get_success_url(self):
        return reverse('contato-sent')

    def get_body_template(self):
        if self.body_template:
            template = self.body_template
        else:
            raise ImproperlyConfigured('Please provide a body_template')
        return template

    def get_subject_template(self):
        if self.subject_template:
            template = self.subject_template
        else:
            raise ImproperlyConfigured('Please provide a subject_template')
        return template

    def get_recipient_list(self):
        if self.recipient_list:
            recipient_list = self.recipient_list
        else:
            try:
                recipient_list = settings.CONTATO_RECIPIENT_LIST
            except AttributeError:
                raise ImproperlyConfigured('Please provide a recipient_list')
        return recipient_list

    def get_from_email(self, form_data):
        raise NotImplementedError

    def get_subject(self, form_data):
        raise NotImplementedError

    def get_body(self, form_data):
        raise NotImplementedError

    def get_headers(self, form_data):
        raise NotImplementedError

    def create_email(self, form_data):
        recipient_list = self.get_recipient_list()
        from_email = self.get_from_email(form_data)
        subject = self.get_subject(form_data)
        body = self.get_body(form_data)
        headers = self.get_headers(form_data)
        email = EmailMessage(subject, body, from_email, recipient_list, headers=headers)
        return email

    def send(self, form_data, fail_silently=False):
        email = self.create_email(form_data)
        email.send(fail_silently=fail_silently)

    def form_valid(self, form):
        try:
            self.send(form.cleaned_data)
        except BadHeaderError:
            return HttpResponse('Invalid header found.')
        return super(BaseContactView, self).form_valid(form)


class ContactView(BaseContactView):
    form_class = ContactForm
    template_name = 'contato/contato_form.html'
    body_template = 'contato/body.txt'
    subject_template = 'contato/subject.txt'
    site_name = None

    def get_site_name(self):
        if self.site_name:
            site_name = self.site_name
        else:
            site_name = get_current_site(self.request).name
        return site_name

    def get_from_email(self, form_data):
        sender_name = form_data.get('name')
        from_email = settings.DEFAULT_FROM_EMAIL
        return '%s <%s>' % (sender_name, from_email)

    def get_subject(self, form_data):
        subject = _('Message from the website contact form')
        site_name = self.get_site_name()
        template = self.get_subject_template()
        string = render_to_string(template, {
            'subject': subject,
            'site_name': site_name,
            'form_data': form_data
        })
        return string.strip()

    def get_body(self, form_data):
        message = form_data.get('message')
        site_name = self.get_site_name()
        template = self.get_body_template()
        string = render_to_string(template, {
            'message': message,
            'site_name': site_name,
            'form_data': form_data,
        })
        return string.lstrip()

    def get_headers(self, form_data):
        sender_name = form_data.get('name')
        sender_email = form_data.get('email')
        reply_to = '%s <%s>' % (sender_name, sender_email)
        return {'Reply-To': reply_to}


class AntiSpamContactView(ContactView):
    form_class = AntiSpamContactForm
