"""
Memcached cache backend for Django using pylibmc.

If you want to use the binary protocol, specify `'BINARY': True` in your CACHES
settings.  The default is `False`, using the text protocol.

pylibmc behaviors can be declared as a dict in `CACHES` backend `OPTIONS`
setting.

Unlike the default Django caching backends, this backend lets you pass 0 as a
timeout, which translates to an infinite timeout in memcached.
"""
import logging
import warnings
from threading import local

from django.conf import settings
from django.core.cache.backends.base import InvalidCacheBackendError
from django.core.cache.backends.memcached import BaseMemcachedCache

from couchbase import Couchbase,connection,exceptions

log = logging.getLogger('django.couchbase')

class CouchbaseCache(BaseMemcachedCache):
    def __init__(self, server, params, username=None, password=None):
        import os
        self._server = server
        self._client = None
        self._params = params
        super(CouchbaseCache, self).__init__(self._server, params, library=Couchbase,
                                           value_not_found_exception=ValueError)

    @property
    def _cache(self):
        # PylibMC uses cache options as the 'behaviors' attribute.
        # It also needs to use threadlocals, because some versions of
        # PylibMC don't play well with the GIL.
        client = self._client
        if client:
            return client
        host = None
        port = 8091
        if len(self._server) > 0:
            idx = self._server[0].find(':')
            host = self._server[0][:idx]
            port = self._server[0][idx+1:]

        if host == None:
            raise Exception('need host')

        client = self._lib.connect(  bucket= self._options.get('bucket', 'default'),
                                     host=host,
                                     password=self._options.get('password', ''),
                                     port=port,
                                     timeout=self._options.get('operation_timeout', 10 ),
                                     lockmode=connection.LOCKMODE_WAIT )

        self._client = client

        return client


    def get(self, key, default=None, version=None):
        try:
            rs = super(CouchbaseCache, self).get(key, default, version)
            return rs.value
        except Exception, e:
            #log.error('CouchbaseError: %s' % e, exc_info=True)
            return default



    def set(self, key, value, timeout=None, version=None):
        key = self.make_key(key, version=version)
        cn = self._cache
        rs = None
        
        try:
            cn.lock( key,  ttl=self._get_memcache_timeout(timeout) )
            
            rs = cn.set(key, value,
                         ttl=self._get_memcache_timeout(timeout)
                        )
        except exceptions.KeyExistsError,e:
            #pass
            #cn.replace( key, value, ttl=self._get_memcache_timeout(timeout) ) 
            log.error( str(e) )
        except exceptions.NotFoundError:
            #pass
            #log.error('ServerError saving %s (%d bytes)' % (key, len(value)),
            #          exc_info=True)
            
            rs = self._cache.add(key, value, ttl=self._get_memcache_timeout(timeout) )
        except Exception, e:
            log.error('MemcachedError: %s' % e, exc_info=True)
            rs = False

        return rs

    def add(self, key, value, timeout=0, version=None):
        key = self.make_key(key, version=version) 
        rs = self._cache.add(key, value, ttl=self._get_memcache_timeout(timeout) )
        return rs

    def close(self, **kwargs):
        pass
