import os
from django.core.files.storage import Storage
from celery import task
from .signals import transaction_signals


@task
def delete_file(filename):
    os.remove(filename)


class DelayHook(Storage):
    def __init__(self, storage):
        self.storage = storage
        self.reset()

    def reset(self):
        self.delete_on_commit = []
        self.delete_on_rollback = []

    def delete(self, name):
        self.delete_on_commit.append(name)

    def _save(self, name, content):
        self.delete_on_rollback.append(name)
        return self.storage.save(name, content)

    def _open(self, name, mode):
        self.storage.open(name, mode)

    def exists(self, name):
        return self.storage.exists(name)

    def on_commit(self):
        for f in self.delete_on_commit:
            delete_file.delay(f)
        self.reset()

    def on_rollback(self):
        for f in self.delete_on_rollback:
            delete_file.delay(f)
        self.reset()


def signal_processor(storage):
    def on_commit(signal, **kwargs):
        storage.on_commit()

    def on_rollback(signal, **kwargs):
        storage.on_rollback()

    transaction_signals.post_commit.connect(on_commit, weak=False)
    transaction_signals.post_rollback.connect(on_rollback, weak=False)
