import hashlib
import logging
import simplejson as json
import xmltodict

from datetime import datetime

from haystack.views import SearchView
from haystack.query import SearchQuerySet

from django.http import HttpResponse
from django.shortcuts import render_to_response
from django.template import loader
from django.conf import settings
from django.views.decorators.csrf import csrf_exempt
from django.core.exceptions import PermissionDenied
from django.db.models import Count

from .forms import DictionaryUploadForm
from .utils import handle_xml_upload
from .models import VisitDetails, SearchEntry


class XMLResponseSearchView(SearchView):
    """Implements a standard SearchView which returns an HttpResponse with
    content type = text/xml."""
    # TODO: Decide whether this should be considered a mixin.

    def create_response(self):
        """
        Generates the actual HttpResponse to send back to the user.
        """
        (paginator, page) = self.build_page()
        context = {
            'query': self.query,
            'form': self.form,
            'page': page,
            'paginator': paginator,
            'suggestion': None,
        }

        logger = logging.getLogger(settings.DICTIONARY_LOGGER)
        # TODO: Create log entry her
        query = self.query
        now = datetime.now()
        ua = self.request.GET.get('user_agent',
                                  self.request.META['HTTP_USER_AGENT'])
        ra = self.request.GET.get('ip', self.request.META['REMOTE_ADDR'])
        ru = self.request.GET.get('from_url', '')
        referer = self.request.GET.get('referer', '')
        count = self.form.sqs.count()
        # Log search here:
        visit_details = VisitDetails(
            query=query,
            user_agent=ua,
            remote_ip=ra,
            calling_url=ru,
            referer=referer,
            count=count
        )
        visit_details.save()
 
        context.update(self.extra_context())
        # This is the payload where we specify the return type

        as_json = self.request.GET.get('as_json', 0)
        
        if as_json:
            # First, render XML
            xml = loader.render_to_string(
                self.template,
                context,
                context_instance=self.context_class(self.request))
            # Convert to JSON
            # TODO: This is not good enough - it's fine for other fields, but
            # it breaks the embedded HTML.
            dict_list = xmltodict.parse(xml)
            js = json.dumps(dict_list)
            # Return as HttpResponse
            return HttpResponse(js, content_type='application/json')

        else:
            return render_to_response(
                self.template,
                context,
                context_instance=self.context_class(self.request),
                content_type='text/xml')


def autocomplete(request):
    query = request.GET.get('q', '')
    sqs = SearchQuerySet()
    sqs = (
        sqs.filter(content_auto__startswith=query)
    )
    # Performance: Don't get a longer list than necessary
    prefix_suggestions = [
        (result.entry, result.pk, result.chronology) for result in sqs if
        result.entry.startswith(query)
    ][:settings.RESULTS_PER_PAGE]

    # PHILIP: sort prefix_suggestions by year
    prefix_suggestions = sorted(prefix_suggestions, key=lambda x: x[2], reverse=True)   

    if len(prefix_suggestions) < settings.RESULTS_PER_PAGE:
        other_suggestions = [
            (result.entry, result.pk, result.chronology) for result in sqs if 
            query in result.entry and not result.entry.startswith(query)
        ][:settings.RESULTS_PER_PAGE]
    else:
        other_suggestions = []

    suggestions = (prefix_suggestions +
                   other_suggestions)[:settings.RESULTS_PER_PAGE]

    # Now get total results count.
    # XXX WARNING: Potential performance hit.
    sqs = SearchQuerySet()

    sqs = (sqs.filter(entry__exact=query) |
           sqs.filter(entry__startswith=query) |
           sqs.filter(text__startswith=query))
    count = sqs.count()
    # Make sure you return a JSON object, not a bare list.
    # Otherwise, you could be vulnerable to an XSS attack.
    the_data = json.dumps({
        'results': suggestions, 'total': count
    })
    return HttpResponse(the_data, content_type='application/json')


def most_popular(request):
    queries_with_count = VisitDetails.objects.values('query').annotate(
        itemcount=Count('query')
    )
    most_popular_queries = queries_with_count.order_by(
        '-itemcount'
    )[:5 * settings.RESULTS_PER_PAGE]

    most_popular_words = [
        (pq['query'], pq['itemcount']) for pq in most_popular_queries if
         len(SearchEntry.objects.filter(
             entry_normalized=pq['query']
         )) > 0][:settings.RESULTS_PER_PAGE]

    return HttpResponse(most_popular_words, content_type='application/json')


def random_json(request):
    random_entries = SearchEntry.objects.order_by(
        '?'
    )[:settings.RESULTS_PER_PAGE]
    words = [se.entry for se in random_entries]
    js = json.dumps(words)
    return HttpResponse(js, content_type='application/json')


@csrf_exempt
def dictionary_upload(request):
    """Handle upload of XML dictionary."""
    if request.method == 'POST':
        form = DictionaryUploadForm(request.POST, request.FILES)
        print request.FILES
        if form.is_valid():
            # Check password
            sha = hashlib.sha1()
            sha.update(request.POST['upload_password'])
            given_password = sha.hexdigest()
            if given_password != settings.UPLOAD_PASSWORD:
                raise PermissionDenied
            # Perform the import
            xml_file = request.FILES['xml_file']
            try:
                n_records_imported = handle_xml_upload(xml_file)
            except:
                raise
            if n_records_imported > 0:
                return HttpResponse(
                    'OK - {0} records indexed'.format(n_records_imported)
                )
            else:
                return HttpResponse('No records found - no changes made.')
    else:
        form = DictionaryUploadForm()

    return render_to_response(
        'dictionary_search/dictionary_upload.html',
        {'form': form}
    )
