# Miscellaneous utility functions.

import sys
from logging import Handler
from xml.etree import ElementTree as ET

from django.core.management import call_command

from dictionary_search.models import SearchEntry


def handle_xml_upload(xml_file):
    """Perform the import of an XML file from the user.

    The xml_file parameter is an UploadedFile and has several attributes, which
    include a read() function and a name.
    """
    xml_data = xml_file.read()
    # Import to Django database
    num_entries = do_import_xml(xml_data)

    # Rebuild index in backend
    # No output - if this fails, an exception is thrown
    if num_entries > 0:
        call_command('rebuild_index', interactive=False, verbosity=0)

    return num_entries


def do_import_xml(xml_data, silent=True, dry=False):

    root = ET.fromstring(xml_data)
    entries = root.findall('entry')
    if not silent:
        sys.stdout.write("\n")
        sys.stdout.write("{0} entries found.\n".format(len(entries)))
        # Data parsed OK - now delete existing entries and reimport everything.
        response = raw_input(
            "WARNING: Deleting existing data before import. Proceed? (y/N)"
        )
        if response != 'y':
            sys.stderr.write("Exiting ...")
            sys.exit()

    # Delete existing entries
    if not dry and len(entries) > 0:
        SearchEntry.objects.all().delete()
    # Do the import.
    for entry in entries:
        entry_values = {}
        for child in entry:
            entry_values[child.tag] = child
        # Utility function to extract XML content of tag
        as_xml = lambda t: ET.tostring(entry_values[t]).replace(
            '<' + t + '>', '').replace('</' + t + '>', '').strip()
        # Utility to extract text value of tag
        as_text = lambda t: entry_values[t].text or ''
        normalize = lambda q: q.strip("(-)").split('#')[0]

        new_se = SearchEntry()
        new_se.entry = as_text('alfa')
        new_se.entry_normalized = normalize(new_se.entry)
        new_se.chronology = as_text('krono')
        new_se.word_class = as_text('ordklasse')
        new_se.language = as_text('sprog')
        new_se.priority_level_1 = as_text('prio1')
        new_se.priority_level_2 = as_text('prio2')
        new_se.priority_level_3 = as_text('prio3')
        new_se.priority_level_4 = as_text('prio4')
        new_se.snippet_text = as_xml('backdoor_lemma')
        new_se.snippet_link = as_text('backdoor_link')
        new_se.html = as_xml('html')

        if not dry:
            new_se.save()

    return len(entries)


class LogHandler(Handler):

    def __init__(level):
        Handler.__init__(level)

    def handle(self, log_record):
        print "LOG: " + str(log_record)
