from django.test import TestCase

from django.contrib.auth.models import User

from django_dnd.backends import DNDBackend
from django_dnd.models import DND

class TestDndBackend(TestCase):
    
    def setUp(self):
        self.backend = DNDBackend()
        self.user = User.objects.create_user('Peter J. Gadomski@DARTMOUTH.EDU', '')
    
    def test_no_user(self):
        user = self.backend.ensure_dnd(None)
        assert user is None
    
    def test_dnd_already_exists(self):
        DND.objects.create(user=self.user, uid='123')
        assert self.backend.ensure_dnd(self.user) == self.user
    
    def test_no_dnd(self):
        user = self.backend.ensure_dnd(self.user)
        user = User.objects.get(pk=user.pk)
        assert user == self.user
        assert user.first_name == 'Peter'
        assert user.last_name == 'Gadomski'
        assert user.email == 'Peter.J.Gadomski@Dartmouth.edu', 'email is %s' % user.email
        assert user.username == 'Peter J. Gadomski@DARTMOUTH.EDU'
        assert user.dnd.uid == 218565977, user.dnd.uid
    
    def test_not_an_email_username(self):
        "When the username is not an email, assume its not a DND login"
        self.user.username = 'peteg'
        assert self.backend.ensure_dnd(self.user) == self.user
    
    def test_no_user_found(self):
        self.user.username = 'PANDABEARSAREAWESOME@DARTMOUTH.EDU'
        assert self.backend.ensure_dnd(self.user) == self.user
    
    def test_too_many_users(self):
        self.user.username = 'peter@DARTMOUTH.EDU'
        assert self.backend.ensure_dnd(self.user) == self.user
    
    def test_different_name(self):
        olduser = User.objects.create_user('Peter Gadomski@DARTMOUTH.EDU', '')
        self.backend.ensure_dnd(olduser)
        
        user = self.backend.ensure_dnd(self.user)
        user = User.objects.get(pk=user.pk)
        assert user == olduser
        assert user.username == 'Peter J. Gadomski@DARTMOUTH.EDU'
