"""DND authentication backend"""

from django.conf import settings
from django.db import IntegrityError

from django_cas.backends import CASBackend

from models import DND

LOOKUP_FIELDS = ('uid', 'firstname', 'lastname', 'email')

class DNDBackend(CASBackend):
    """DND authentication backend"""
    
    def authenticate(self, ticket, service):
        """
        Verify the CAS ticket with django_cas, then create the DND if needed.
        
        """
        user = super(DNDBackend, self).authenticate(ticket, service)
        return self.ensure_dnd(user)
    
    def ensure_dnd(self, user):
        """
        Ensure that a DND entry exists for the given user.
        If a new user was created by django_cas, but the uid for that username
        already exists, delete the new user and change the username of the old
        user.
        
        """
        if not user:
            return None
        if '@' not in user.username:
            return user
        try:
            return DND.objects.get(user=user).user
        except DND.DoesNotExist:
            pass
        
        import dnd
        name = user.username.split('@')[0]
        with dnd.DNDSession(server=settings.DND_SERVER) as db:
            try:
                result = db.lookup_unique(name, *LOOKUP_FIELDS)
            except dnd.DNDProtocolError: # No user found
                return user
        if not result: # Too many users found
            return user
        try:
            dnd = DND.objects.create(uid=result['uid'], user=user)
            user.first_name = result['firstname']
            user.last_name = result['lastname']
            user.email = result['email']
            user.save()
        except IntegrityError:
            dnd = DND.objects.get(uid=result['uid'])
            dnd.user.username = user.username
            user.delete()
            dnd.user.save()
            user = dnd.user
        return user
    

        