from django.conf import settings
from django.core.mail import EmailMultiAlternatives, SafeMIMEMultipart
from django.core.mail.message import make_msgid
from email.MIMEBase import MIMEBase
from email.utils import formatdate
from gpg import detach_sign, encrypt

class EmailPGP(EmailMultiAlternatives):
	encrypted_subtype = 'encrypted'
	mixed_subtype = 'mixed'
	signed_subtype = 'signed'
	
	def __init__(self, subject='', body='', from_email=None, to=None, bcc=None,
		connection=None, attachments=None, headers=None, alternatives=None,
		encrypted=False, signed=False):
		
		self.encrypted = encrypted
		self.signed = signed
		
		return super(EmailPGP, self).__init__(subject, body, from_email, to, bcc, connection, attachments, headers, alternatives)

	def message(self):
		msg=super(EmailPGP, self).message()
		encoding = self.encoding or settings.DEFAULT_CHARSET
		del msg['From']
		del msg['Subject']
		del msg['To']
		del msg['Date']
		
		if self.signed:
			tmp = SafeMIMEMultipart(_subtype=self.signed_subtype, encoding=encoding)
			tmp.attach(msg)
			attachment = MIMEBase('application', 'pgp-signature') #We don't want base64 enconding
			attachment.set_payload(detach_sign(msg.as_string(), self.from_email))
			attachment.add_header('Content-Disposition', 'attachment', filename='signature.asc')
			tmp.attach(attachment)
			msg=tmp
		
		if self.encrypted:
			tmp = SafeMIMEMultipart(_subtype=self.encrypted_subtype, encoding=encoding)
			tmp.attach(self._create_attachment('', '', 'application/pgp-encrypted'))
			attachment = MIMEBase('application', 'octet-stream') #We don't want base64 enconding
			attachment.set_payload(encrypt(msg.as_string(), self.to))
			attachment.add_header('Content-Disposition', 'inline', filename='msg.asc')
			tmp.attach(attachment)
			msg=tmp
		
		msg['Subject'] = self.subject
		msg['From'] = self.extra_headers.get('From', self.from_email)
		msg['To'] = self.extra_headers.get('To', ', '.join(self.to))
		if self.cc:
			msg['Cc'] = ', '.join(self.cc)
		# Email header names are case-insensitive (RFC 2045), so we have to
		# accommodate that when doing comparisons.
		header_names = [key.lower() for key in self.extra_headers]
		if 'date' not in header_names:
			msg['Date'] = formatdate()
		if 'message-id' not in header_names:
			msg['Message-ID'] = make_msgid()
		for name, value in self.extra_headers.items():
			if name.lower() in ('from', 'to'):  # From and To are already handled
				continue
			msg[name] = value
		return msg
