from django.conf import settings
from django.core.mail import get_connection
from EmailPGP import EmailPGP

def send_mail(subject, message, from_email, recipient_list,
		fail_silently=False, auth_user=None, auth_password=None,
		connection=None, html_message=None, attachments=None,
		encrypted=False, signed=False):
	"""
	Easy wrapper for sending a single encrypted and/or signed message 
	(html and/or plain text) eith attachments to a recipient list. 
	All members of the recipient list will see the other recipients 
	in the 'To' field.

	If auth_user is None, the EMAIL_HOST_USER setting is used.
	If auth_password is None, the EMAIL_HOST_PASSWORD setting is used.

	Note: The API for this method is frozen. New code wanting to extend the
	functionality should use the EmailMessage class directly.
	"""
	connection = connection or get_connection(username=auth_user,
		password=auth_password, fail_silently=fail_silently)
	
	mail=EmailPGP(subject, message, from_email, recipient_list, connection=connection, encrypted=encrypted, signed=signed)
	
	if html_message:
		mail.attach_alternative(html_message, "text/html")
	
	for attachment in attachments:
		mail.attach(*attachment)
	
	return mail.send(fail_silently=fail_silently)

def send_mass_mail(datatuple, fail_silently=False, auth_user=None,
		auth_password=None, connection=None, encrypted=False, signed=False):
	"""
	Given a datatuple of (subject, message, from_email, recipient_list, html_message, attachments), sends
	each encrypted and/or signed message to each recipient list. Returns the number of emails sent.
	
	If from_email is None, the DEFAULT_FROM_EMAIL setting is used.
	If auth_user and auth_password are set, they're used to log in.
	If auth_user is None, the EMAIL_HOST_USER setting is used.
	If auth_password is None, the EMAIL_HOST_PASSWORD setting is used.

	Note: The API for this method is frozen. New code wanting to extend the
	functionality should use the EmailMessage class directly.
	"""
	connection = connection or get_connection(username=auth_user, password=auth_password, fail_silently=fail_silently)
	messages = []
	for line in datatuple:
		tmp=EmailPGP(line[0], line[1], line[2], line[3], encrypted=encrypted, signed=signed)
		if line[4]:
			tmp.attach_alternative(line[4], "text/html")
		for attachment in line[5]:
			tmp.attach(*attachment)
		messages.append(tmp)
	return connection.send_messages(messages)

def mail_admins(subject, message, fail_silently=False, connection=None,
		html_message=None, attachments=None,
		encrypted=False, signed=False):
	"""Sends an encrypted and/or signed message with attachments to the admins, as defined by the ADMINS setting."""
	if not settings.ADMINS:
		return
	mail = EmailPGP(u'%s%s' % (settings.EMAIL_SUBJECT_PREFIX, subject),
		message, settings.SERVER_EMAIL, [a[1] for a in settings.ADMINS],
		connection=connection, encrypted=encrypted, signed=signed)
	if html_message:
		mail.attach_alternative(html_message, 'text/html')
	for attachment in attachments:
		mail.attach(*attachment)
	return mail.send(fail_silently=fail_silently)

def mail_managers(subject, message, fail_silently=False, connection=None,
		html_message=None, attachments=None,
		encrypted=False, signed=False):
	"""Sends an encrypted and/or signed message with attachments to the managers, as defined by the MANAGERS setting."""
	if not settings.MANAGERS:
		return
	mail = EmailPGP(u'%s%s' % (settings.EMAIL_SUBJECT_PREFIX, subject),
		message, settings.SERVER_EMAIL, [a[1] for a in settings.MANAGERS],
		connection=connection, encrypted=encrypted, signed=signed)
	if html_message:
		mail.attach_alternative(html_message, 'text/html')
	for attachment in attachments:
		mail.attach(*attachment)
	return mail.send(fail_silently=fail_silently)
