from cms.plugin_base import CMSPluginBase
from cms.plugin_pool import plugin_pool
from django.utils.translation import ugettext_lazy as _
from models import HousesOverview


class AbstractOverviewPlugin(CMSPluginBase):
    model = HousesOverview

    def _get_thumbnail_options(self, context, instance):
        """
        Return the size and options of the thumbnail that should be inserted
        """
        width, height = None, None
        placeholder_width = context.get('width', None)
        placeholder_height = context.get('height', None)
        if instance.use_autoscale and placeholder_width:
            # use the placeholder width as a hint for sizing
            width = int(placeholder_width)
        if instance.use_autoscale and placeholder_height:
            height = int(placeholder_height)
        elif instance.width:
            width = instance.width
        if instance.height:
            height = instance.height
        if instance.overview.map:
            if not height and width:
                # height was not externally defined: use ratio to scale it by the width
                height = int(float(width)*float(instance.overview.map.height)/float(instance.overview.map.width))
            if not width and height:
                # width was not externally defined: use ratio to scale it by the height
                width = int(float(height)*float(instance.overview.map.width)/float(instance.overview.map.height))
            if not width:
                # width is still not defined. fallback the actual map width
                width = instance.overview.map.width
            if not height:
                # height is still not defined. fallback the actual map height
                height = instance.overview.map.height
        return {'size': (width, height), 'ratio': float(width)/instance.overview.map.width}

    def render(self, context, instance, placeholder):
        context = super(AbstractOverviewPlugin, self).render(context, instance, placeholder)
        options = self._get_thumbnail_options(context, instance)
        context.update({
            'instance': instance,
            'opts': options,
            'size': options.get('size', None),
            'ratio': str(options.get('ratio', 1)).replace(",", "."),
            'overlays': instance.overview.overlay_set.all,
        })
        return context


class HousesOverviewPlugin(AbstractOverviewPlugin):
    name = _("Houses - Overview")
    render_template = "realestate_maps/plugins/housesoverview.html"

plugin_pool.register_plugin(HousesOverviewPlugin)