from django.db import models
from django.utils.translation import ugettext_lazy as _


class Group(models.Model):
    name = models.CharField(_('name'), max_length=50, blank=False)
    website = models.CharField(_('website'), max_length=50, blank=True)
    longitude = models.CharField(_('longitude'), max_length=50, blank=True, null=True)
    latitude = models.CharField(_('latitude'), max_length=50, blank=True, null=True)

    def __unicode__(self):
        return self.name

    def available_houses(self):
        return self.house_group.filter(status=House.FOR_SALE)

    class Meta:
        verbose_name = _('project')
        verbose_name_plural = _('projects')


class SubGroup(models.Model):
    group = models.ForeignKey(Group, null=False, blank=False, related_name='subgroup_group',
                              verbose_name=_('project'), on_delete=models.CASCADE)
    name = models.CharField(_('name'), max_length=50, blank=False)

    def __unicode__(self):
        return self.group.__unicode__()+' - '+self.name

    class Meta:
        verbose_name = _('house type')
        verbose_name_plural = _('house types')


class Fase(models.Model):
    name = models.CharField(_('name'), max_length=50, blank=True)
    ready = models.DateField(_('ready'), null=True, blank=True)
    start_building = models.DateField(_('start building'), null=True, blank=True)
    group = models.ForeignKey(Group, verbose_name=_('project'), on_delete=models.CASCADE)

    class Meta:
        verbose_name = _('fase')
        verbose_name_plural = _('fases')

    def __unicode__(self):
        return self.group.__unicode__()+' '+self.name


class HouseKind(models.Model):
    name = models.CharField(_('name'), max_length=50, blank=True)

    def __unicode__(self):
        return self.name

    class Meta:
        verbose_name = _('kind of house')
        verbose_name_plural = _('kinds of houses')


class HouseType(models.Model):
    parent = models.ForeignKey(Group, null=False, related_name="housetype_group", verbose_name=_('project'),
                               on_delete=models.CASCADE)
    subgroup = models.ForeignKey(SubGroup, null=True, blank=True, related_name='housetype_subgroup',
                                 verbose_name=_('house type'), on_delete=models.CASCADE)
    name = models.CharField(_('name'), max_length=50, blank=False)
    rooms = models.IntegerField(_('rooms'), null=True, blank=True)
    living_area = models.IntegerField(_('living area'), null=True, blank=True)
    ground_area = models.IntegerField(_('ground area'), null=True, blank=True)
    bvo = models.IntegerField(_('bvo'), null=True, blank=True)
    outdoor_space = models.IntegerField(_('outdoor space'), null=True, blank=True)
    outdoor_storage = models.IntegerField(_('outdoor storage'), null=True, blank=True)
    base_price = models.IntegerField(_('base price'), null=True, blank=True)
    house_kinds = models.ManyToManyField(HouseKind, blank=True, verbose_name=_('house category'))

    def __unicode__(self):
        return self.name

    class Meta:
        verbose_name = _('house specification')
        verbose_name_plural = _('house specifications')


class House(models.Model):
    NORTH = 'N'
    EAST = 'E'
    SOUTH = 'S'
    WEST = 'W'
    NORTH_EAST = 'NE'
    SOUTH_EAST = 'SE'
    SOUTH_WEST = 'SW'
    NORTH_WEST = 'NW'
    GARDEN_CHOICES = ((NORTH, _('north')), (EAST, _('east')),
                      (SOUTH, _('south')), (WEST, _('west')),
                      (NORTH_EAST, _('north-east')), (NORTH_WEST, _('south-east')),
                      (SOUTH_WEST, _('south-west')), (NORTH_WEST, _('north-west')))

    FOR_SALE = 'FS'
    OPTIONED = 'OP'
    SOLD = 'SO'
    STATUS_CHOICES = ((FOR_SALE, _('for sale')), (OPTIONED, _('optioned')), (SOLD, _('sold')))

    fase = models.ForeignKey(Fase, blank=True, null=True, verbose_name=_('fase'), on_delete=models.SET_NULL)
    # TODO On delete, set to parent's parent
    parent = models.ForeignKey(Group, null=False, related_name="house_group", verbose_name=_('project'),
                               on_delete=models.CASCADE)
    subgroup = models.ForeignKey(SubGroup, null=True, blank=True, related_name='house_subgroup',
                                 verbose_name=_('house type'), on_delete=models.CASCADE)
    house_type = models.ForeignKey(HouseType, blank=False, related_name="house_house_type",
                                   verbose_name=_('house specification'), on_delete=models.CASCADE)
    number = models.IntegerField(_('number'), null=True, blank=False)
    garden = models.CharField(_('garden'), max_length=2, choices=GARDEN_CHOICES, null=True, blank=True)
    status = models.CharField(_('status'), max_length=2, default=FOR_SALE, choices=STATUS_CHOICES)

    def __unicode__(self):
        return self.parent.__unicode__() + ' - ' + self.number.__str__()

    class Meta:
        verbose_name = _('house')
        verbose_name_plural = _('houses')
        ordering = ['number']