"""
Internal module for the plugin system,
the API is exposed via __init__.py
"""
from threading import Lock
from django.conf import settings
from django.contrib.contenttypes.models import ContentType
from django.utils.importlib import import_module
from fluent_contents.forms import ContentItemForm
from fluent_contents.models import ContentItem
from .pluginbase import ContentPlugin


__all__ = ('PluginContext', 'ContentPlugin', 'plugin_pool')


# This mechanism is mostly inspired by Django CMS,
# which nice job at defining a clear extension model.
# (c) Django CMS developers, BSD licensed.

# Some standard request processors to use in the plugins,
# Naturally, you want STATIC_URL to be available in plugins.


def _import_apps_submodule(submodule):
    """
    Look for a submodule is a series of packages, e.g. ".content_plugins" in all INSTALLED_APPS.
    """
    for app in settings.INSTALLED_APPS:
        try:
            import_module('.' + submodule, app)
        except ImportError, e:
            if submodule not in str(e):
                raise   # import error is a level deeper.
            else:
                pass


class PluginAlreadyRegistered(Exception):
    """
    Raised when attemping to register a plugin twice.
    """
    pass


class PluginNotFound(Exception):
    """
    Raised when the plugin could not be found in the rendering process.
    """
    pass


class PluginPool(object):
    """
    The central administration of plugins.
    """
    scanLock = Lock()

    def __init__(self):
        self.plugins = {}
        self._name_for_model = {}
        self._name_for_ctype_id = None
        self.detected = False


    def register(self, plugin):
        """
        Make a plugin known by the CMS.

        :param plugin: The plugin class, deriving from :class:`ContentPlugin`.

        The plugin will be instantiated, just like Django does this with :class:`~django.contrib.admin.ModelAdmin` classes.
        If a plugin is already registered, this will raise a :class:`PluginAlreadyRegistered` exception.
        """
        # Duck-Typing does not suffice here, avoid hard to debug problems by upfront checks.
        assert issubclass(plugin, ContentPlugin), "The plugin must inherit from `ContentPlugin`"
        assert plugin.model, "The plugin has no model defined"
        assert issubclass(plugin.model, ContentItem), "The plugin model must inherit from `ContentItem`"
        assert issubclass(plugin.form, ContentItemForm), "The plugin form must inherit from `ContentItemForm`"

        name = plugin.__name__
        if name in self.plugins:
            raise PluginAlreadyRegistered("{0}: a plugin with this name is already registered".format(name))
        name = name.lower()

        # Make a single static instance, similar to ModelAdmin.
        plugin_instance = plugin()
        self.plugins[name] = plugin_instance
        self._name_for_model[plugin.model] = name       # Track reverse for model.plugin link

        # Only update lazy indexes if already created
        if self._name_for_ctype_id is not None:
            self._name_for_ctype_id[plugin.type_id] = name

        return plugin  # Allow decorator syntax


    def get_plugins(self):
        """
        Return the list of all plugin instances which are loaded.
        """
        self._import_plugins()
        return self.plugins.values()


    def get_plugins_by_name(self, *names):
        """
        Return a list of plugins by plugin class, or name.
        """
        self._import_plugins()
        plugin_instances = []
        for name in names:
            if isinstance(name, basestring):
                try:
                    plugin_instances.append(self.plugins[name.lower()])
                except KeyError:
                    raise PluginNotFound("No plugin named '{0}'.".format(name))
            elif issubclass(name, ContentPlugin):
                # Will also allow classes instead of strings.
                plugin_instances.append(self.plugins[self._name_for_model[name.model]])
            else:
                raise TypeError("get_plugins_by_name() expects a plugin name or class, not: {0}".format(name))
        return plugin_instances


    def get_model_classes(self):
        """
        Return all :class:`~fluent_contents.models.ContentItem` model classes which are exposed by plugins.
        """
        self._import_plugins()
        return [plugin.model for plugin in self.plugins.values()]


    def get_plugin_by_model(self, model_class):
        """
        Return the corresponding plugin for a given model.
        """
        self._import_plugins()                       # could happen during rendering that no plugin scan happened yet.
        assert issubclass(model_class, ContentItem)  # avoid confusion between model instance and class here!

        try:
            name = self._name_for_model[model_class]
        except KeyError:
            raise PluginNotFound("No plugin found for model '{0}'.".format(model_class.__name__))
        return self.plugins[name]


    def _get_plugin_by_content_type(self, contenttype):
        self._import_plugins()
        self._setup_lazy_indexes()

        ct_id = contenttype.id if isinstance(contenttype, ContentType) else int(contenttype)
        try:
            name = self._name_for_ctype_id[ct_id]
        except KeyError:
            raise PluginNotFound("No plugin found for content type '{0}'.".format(contenttype))
        return self.plugins[name]


    def _import_plugins(self):
        """
        Internal function, ensure all plugin packages are imported.
        """
        if self.detected:
            return

        # In some cases, plugin scanning may start during a request.
        # Make sure there is only one thread scanning for plugins.
        self.scanLock.acquire()
        if self.detected:
            return  # previous threaded released + completed

        try:
            _import_apps_submodule("content_plugins")
            self.detected = True
        finally:
            self.scanLock.release()

    def _setup_lazy_indexes(self):
        # The ContentType is not read yet at .register() time, since that enforces the database to exist at that time.
        # If a plugin library is imported via different paths that might not be the case when `./manage.py syncdb` runs.
        if self._name_for_ctype_id is None:
            plugin_ctypes = {}  # separate dict to build, thread safe
            self._import_plugins()
            for name, plugin in self.plugins.iteritems():
                plugin_ctypes[plugin.type_id] = name

            self._name_for_ctype_id = plugin_ctypes


#: The global plugin pool, a instance of the :class:`PluginPool` class.
plugin_pool = PluginPool()
