import smtplib

from datetime import datetime

from django.conf import settings
from django.db import models
from django.utils.translation import ugettext_lazy as _

from informativo.utils import send_mass_mail


NEWSLETTER_STATUS = (
    ('canceled', _('canceled')),
    ('waiting', _('waiting sending')),
    ('sending', _('sending')),
    ('sent', _('sent')),
)

CONTACT_MAILING_STATUS = (
    ('canceled', _('canceled')),
    ('sending', _('sending')),
    ('sent', _('sent')),
    ('failed', _('failed')),
)


class Contact(models.Model):
    email = models.EmailField(_('e-mail'))
    first_name = models.CharField(_('first name'), max_length=64, blank=True)
    last_name = models.CharField(_('last name'), max_length=64, blank=True)
    mailing_lists = models.ManyToManyField(
        'informativo.MailingList',
        verbose_name=_('mailing lists'),
        related_name='subscribers'
    )

    class Meta:
        verbose_name = _('contact')
        verbose_name_plural = _('contacts')
        ordering = ('-id',)

    def __unicode__(self):
        if self.first_name and self.last_name:
            return u'%s %s <%s>' % (self.first_name, self.last_name, self.email)
        return u'%s' % self.email


class MailingList(models.Model):
    name = models.CharField(_('name'), max_length=64)
    description = models.TextField(_('description'), blank=True)

    class Meta:
        verbose_name = _('mailing list')
        verbose_name_plural = _('mailing lists')
        ordering = ('-id',)

    def __unicode__(self):
        return unicode(self.name)


class Newsletter(models.Model):
    title = models.CharField(
        _('title'),
        max_length=128,
        help_text=_("Used on the e-mail's subject")
    )
    html_content = models.TextField(_('HTML content'), blank=True)
    text_content = models.TextField(_('text content'), blank=True)
    mailing_list = models.ForeignKey(MailingList, verbose_name=_('mailing list'))
    sending_date = models.DateTimeField(
        _('sending date'),
        default=datetime.now,
    )
    status = models.CharField(
        _('status'),
        max_length=16,
        choices=NEWSLETTER_STATUS,
        default='waiting'
    )

    class Meta:
        verbose_name = _('newsletter')
        verbose_name_plural = _('newsletters')
        ordering = ('-id',)

    def __unicode__(self):
        return unicode(self.title)

    def send(self):
        subject = self.title
        text_content, html_content = self.text_content, self.html_content
        from_email = settings.DEFAULT_FROM_EMAIL

        self.status = 'sending'
        self.save()

        for recipient in self.mailing_list.subscribers.all():
            contact_status, created = ContactMailingStatus.objects.get_or_create(
                contact=recipient,
                newsletter=self,
                defaults={'status': 'sending'}
            )

            # Do not send the same newsletter to the same recipient twice
            if contact_status.status in ('canceled', 'sent'):
                continue

            # Send mail
            datatuple = ((subject, text_content, html_content, from_email, [recipient.email]),)
            try:
                send_mass_mail(datatuple)
                contact_status.status = 'sent'
                status = 'sent'
            except smtplib.SMTPException:
                contact_status.status = 'failed'
            contact_status.save()

        self.status = 'sent'
        self.save()


class ContactMailingStatus(models.Model):
    contact = models.ForeignKey(
        'informativo.Contact',
        verbose_name=_('contact')
    )
    newsletter = models.ForeignKey(
        'informativo.Newsletter',
        verbose_name=_('newsletter')
    )
    status = models.CharField(
        _('status'),
        max_length=16,
        choices=CONTACT_MAILING_STATUS
    )

    class Meta:
        verbose_name = _('contact mailing status')
        verbose_name_plural = _('contact mailing statuses')
        ordering = ('-id',)

    def __unicode__(self):
        status = self.get_status_display()
        return u'"%s" to %s %s' % (self.newsletter, self.contact, status)
