from django import forms
from django.utils.translation import ugettext_lazy as _

from informativo.models import Contact, MailingList


def _get_mailinglists_choices():
    mailing_lists = [(m.pk, m.name) for m in MailingList.objects.all()]
    mailing_lists.insert(0, (0, _('All')))
    return mailing_lists


class SubscribeForm(forms.ModelForm):
    mailing_lists = forms.MultipleChoiceField(
        label=_('Mailing lists'),
        initial=[0],
        widget=forms.CheckboxSelectMultiple
    )

    class Meta:
        model = Contact

    def __init__(self, *args, **kwargs):
        super(SubscribeForm, self).__init__(*args, **kwargs)
        self.fields['mailing_lists'].choices = _get_mailinglists_choices()

    def save(self, commit=True):
        instance = super(SubscribeForm, self).save(commit=False)

        pk_list = self.cleaned_data['mailing_lists']
        if '0' in pk_list:
            mailing_lists = MailingList.objects.all()
        else:
            mailing_lists = MailingList.objects.filter(pk__in=pk_list)

        if commit:
            instance.save()
            instance.mailing_lists = mailing_lists

        return instance


class UnsubscribeForm(forms.Form):
    email = forms.EmailField(label=_('E-mail'), max_length=100)
    mailing_lists = forms.MultipleChoiceField(
        label=_('Mailing lists'),
        initial=[0],
        widget=forms.CheckboxSelectMultiple
    )

    def __init__(self, *args, **kwargs):
        super(UnsubscribeForm, self).__init__(*args, **kwargs)
        self.fields['mailing_lists'].choices = _get_mailinglists_choices()

    def clean_email(self):
        data = self.cleaned_data.get('email')
        try:
            self._contact = Contact.objects.get(email=data)
        except:
            msg = _('This e-mail does not exist in our database.')
            raise forms.ValidationError(msg)
        return data

    def unsubscribe(self):
        email = self.cleaned_data['mailing_lists']
        mailing_lists = self.cleaned_data['mailing_lists']
        if '0' in mailing_lists:
            self._contact.mailing_lists.clear()
        else:
            remove_list = MailingList.objects.exclude(pk__in=mailing_lists)
            self._contact.mailing_lists.remove(*remove_list)
        return self._contact
 
