# -*- coding: utf8 -*-
from __future__ import unicode_literals

import json

from django import template
from django.conf import settings
from django.utils import six

from leaflet import (app_settings, SPATIAL_EXTENT, SRID, PLUGINS, PLUGINS_DEFAULT,
                     PLUGIN_ALL, PLUGIN_FORMS)


register = template.Library()


@register.inclusion_tag('leaflet/css.html')
def leaflet_css(plugins=None):
    """

    :param only_plugins:
    :param exclude_plugins:
    :return:
    """
    plugin_names = _get_plugin_names(plugins)
    return {
        "PLUGINS_CSS": _get_all_resources_for_plugins(plugin_names, 'css'),
    }


@register.inclusion_tag('leaflet/js.html')
def leaflet_js(plugins=None):
    """

    :param only_plugins:
    :param exclude_plugins:
    :return:
    """
    plugin_names = _get_plugin_names(plugins)
    with_forms = PLUGIN_FORMS in plugin_names or PLUGIN_ALL in plugin_names
    return {
        "DEBUG": settings.TEMPLATE_DEBUG,
        "SRID": str(SRID) if SRID else None,
        "PLUGINS_JS": _get_all_resources_for_plugins(plugin_names, 'js'),
        "with_forms": with_forms
    }


@register.inclusion_tag('leaflet/_leaflet_map.html')
def leaflet_map(name, callback=None, fitextent=True, creatediv=True, loadevent='load'):
    """

    :param name:
    :param callback:
    :param fitextent:
    :param creatediv:
    :return:
    """
    extent = None
    if SPATIAL_EXTENT is not None:
        # Leaflet uses [lat, lng]
        xmin, ymin, xmax, ymax = SPATIAL_EXTENT
        extent = (ymin, xmin, ymax, xmax)

    djoptions = dict(
        srid=SRID,
        extent=[extent[:2], extent[2:4]],
        fitextent=fitextent,
        center=app_settings['DEFAULT_CENTER'],
        zoom=app_settings['DEFAULT_ZOOM'],
        layers=[(six.text_type(label), url, attrs) for (label, url, attrs) in app_settings.get('TILES')],
        overlays=[(six.text_type(label), url, attrs) for (label, url, attrs) in app_settings.get('OVERLAYS')],
        attributionprefix=app_settings.get('ATTRIBUTION_PREFIX'),
        scale=app_settings.get('SCALE'),
        minimap=app_settings.get('MINIMAP'),
        resetview=app_settings.get('RESET_VIEW'),
        tilesextent=list(app_settings.get('TILES_EXTENT', []))
    )
    if app_settings['MIN_ZOOM'] is not None:
        djoptions['minzoom'] = app_settings['MIN_ZOOM']
    if app_settings['MAX_ZOOM'] is not None:
        djoptions['maxzoom'] = app_settings['MAX_ZOOM']

    return {
        # templatetag options
        'name': name,
        'loadevents': json.dumps(loadevent.split()),
        'creatediv': creatediv,
        'callback': callback,
        # initialization options
        'djoptions': json.dumps(djoptions),
        # settings
        'NO_GLOBALS': app_settings.get('NO_GLOBALS'),
    }


@register.simple_tag
def leaflet_json_config():
    settings_as_json = app_settings.copy()

    if SPATIAL_EXTENT is not None:
        xmin, ymin, xmax, ymax = settings_as_json.pop('SPATIAL_EXTENT')
        settings_as_json['SPATIAL_EXTENT'] = {'xmin': xmin, 'ymin': ymin,
                                              'xmax': xmax, 'ymax': ymax}

    return json.dumps(settings_as_json)


def _get_plugin_names(plugin_names_from_tag_parameter):
    """
    Returns a list of plugin names, specified in the parameter.
    Used by tags to determine which plugins to include
    :param pluging_names_parameter:
    :return:
    """
    if isinstance(plugin_names_from_tag_parameter, (six.binary_type, six.text_type)):
        names = plugin_names_from_tag_parameter.split(',')
        return [n.strip() for n in names]
    else:
        return [PLUGINS_DEFAULT]


def _get_all_resources_for_plugins(plugin_names, resource_type):
    """
    Returns a list of URLs for the plugins with the specified resource type (js, css, ...)
    :param plugin_names:
    :param resource_type:
    :return:
    """
    result = []
    for plugin_name in plugin_names:
        if plugin_name in PLUGINS:
            result.extend(PLUGINS[plugin_name].get(resource_type, []))

    return result
