# Copyright (c) Siemens AG, 2013
#
# This file is part of MANTIS.  MANTIS is free software: you can
# redistribute it and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either version 2
# of the License, or(at your option) any later version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
# details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc., 51
# Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
#



import re
import logging
import hashlib
import pprint
from collections import deque

from django.core.files.base import ContentFile

from django.utils import timezone

from dingos.core.xml_utils import extract_attributes

from dingos.core.decorators import print_arguments
from dingos.core.utilities import search_by_re_list

import dingos

from dingos import *

from dingos.models import FactDataType, write_large_value

from mantis_core.models import \
    Identifier, FactValue

from mantis_openioc_importer.importer import OpenIOC_Import

OpenIOC_Importer = OpenIOC_Import()

from mantis_core.import_handling import MantisImporter

# Import configuration constants from __init__.py

from mantis_stix_importer import RAW_DATA_TO_DB_FOR_LENGTH_LESS_THAN

logger = logging.getLogger(__name__)


pp = pprint.PrettyPrinter(indent=2)


class STIX_Import:
    """
    A class that implements a DINGOS importer for STIX and CybOX

    A DINGOS importer must implement a function ``xml_import``
    with the following signature::

        def xml_import(self,
                   filepath="",
                   xml_content=None,
                   markings=None,
                   identifier_ns_uri=None,
                   **kwargs):

    - either filepath of an XML file or the xml_content as string
      must be provided
    - a list of DINGOS InfoObjects may be added as 'markings':
      all DINGOS InfoObjects generated by the import will be
      tagged with the markings -- one use case is to provide
      provenance or TLP information.
    - a default identifier namespace can be provided

    The kwargs are not read -- they are present to allow the use of the
    DingoImportCommand class for easy definition of commandline import commands
    (the class passes all command line arguments to the xml_import function, so
    without the **kwargs parameter, an error would occur.

    The ``xml_import`` essentially calls the MANTIS-wrapping of the DINGOS
    XML importer; customization of this XML importer is achieved via hooking
    functions that are also defined in this class: these functions govern
    the behavior of the importer, e.g., they tell it when to extract embedded
    content, how to retrieve identifiers and timestamps, etc.

    The MANTIS/DINGOS xml importer returns then the following structure::


           {'id_and_rev_info': Id and revision info of top-level element of the form
                               {'id': ... , 'timestamp': ...}
            'elt_name': Element name of top-level element
            'dict_repr': Dictionary representation of XML, minus the embedded objects -- for
                         those, an 'idref' reference has been generated
            'embedded_objects': List of embedded objects, as dictionary
                                   {"id_and_revision_info": id and revision info of extracted object,
                                    "elt_name": Element name,
                                    "dict_repr" :  dictionary representation of XML of embedded object
                                   }
            'unprocessed' : List of unprocessed embedded objects (as libxml2 Node object)
                           (e.g. for handover to other importer
            'file_content': Content of imported file (or, if content was passed instead of a file name,
                            the original content)}

    It is then the task of the xml_import function to process the returned data and
    generate InfoObjects with calls to the functions provided by DINGOS via the MANTIS wrapper
    ``mantis_core``. Again, hooking functions are used to configure the creation of InfoObjects
    from the generated dictionary representation.

    """


    def __init__(self, *args, **kwargs):

        # The creation time stamp
        self.create_timestamp = timezone.now()

        self.namespace_dict = {None: DINGOS_NAMESPACE_URI}

        self.default_identifier_ns_uri = None

        # Here, we list the processors for embedded non-STIX/CybOX content.
        # Currently, only OpenIOC is treated.

        self.processors = {'OpenIOC2010': OpenIOC_Import}



    def xml_import(self,
                   filepath="",
                   xml_content=None,
                   markings=None,
                   identifier_ns_uri=None,
                   **kwargs):
        """
         Import a STIX or CybOX xml  from file <filepath> or a string passed as ``xml_content``

         You can furhter provide:

         - a list of InfoObjects as markings with which all generated Information Objects
            will be associated (e.g., in order to provide provenance function)
         - a default identifier namespace URI.

         The kwargs are not read -- they are present to allow the use of the
         DingoImportCommand class for easy definition of commandline import commands
         (the class passes all command line arguments to the xml_import function, so
         without the **kwargs parameter, an error would occur.
         """

        self.default_identifier_ns_uri = identifier_ns_uri

        if not markings:
            markings = []

        # Clear internal state such that same object can be reused for
        # multiple imports.

        self.__init__()


        # Use the generic XML import customized for STIX/CybOX import
        # to turn XML into DingoObjDicts

        import_result = MantisImporter.xml_import(xml_fname=filepath,
                                                  xml_content=xml_content,
                                                  ns_mapping=self.namespace_dict,
                                                  embedded_predicate=self.stix_embedding_pred,
                                                  id_and_revision_extractor=self.id_and_revision_extractor)

        # The MANTIS/DINGOS xml importer returns then the following structure::
        #
        #
        #    {'id_and_rev_info': Id and revision info of top-level element of the form
        #        {'id': ... , 'timestamp': ...}
        #    'elt_name': Element name of top-level element
        #    'dict_repr': Dictionary representation of XML, minus the embedded objects -- for
        #                 those, an 'idref' reference has been generated
        #    'embedded_objects': List of embedded objects, as dictionary
        #                       {"id_and_revision_info": id and revision info of extracted object,
        #                        "elt_name": Element name,
        #                        "dict_repr" :  dictionary representation of XML of embedded object
        #                       }
        #    'unprocessed' : List of unprocessed embedded objects (as libxml2 Node object)
        #                    (e.g. for handover to other importer
        #    'file_content': Content of imported file (or, if content was passed instead of a file name,
        #                                                                                         the original content)}


        # Extract information from import result

        top_level_id_and_rev_info = import_result['id_and_rev_info']
        top_level_elt_name = import_result['elt_name']
        top_level_elt_dict = import_result['dict_repr']
        file_content = import_result['file_content']
        embedded_objects = import_result['embedded_objects']
        unprocessed_list = import_result['unprocessed']


        # We now have the top-level object, the list of embedded objects,
        # and possibly a list of hitherto unprocessed XML nodes.
        # As we shall see below, we have configured the DINGOS
        # XML  importer such that it extracts marking structures
        # as separate objects and tags them with a label
        # 'import_first'. So, what we need to do is to go through
        # the embedded objects and collect all markings (along with
        # information, in which STIX_Package each marking was defined).
        # This allows us to first create all marking objects and
        # then mark the subsequently created objects with the
        # appropriate markings.
        # Note, that currently,
        # - we only extract and treat markings defined in the header of STIX_Package
        # - we do not care about the XPATH-expression and treat every marking as if
        #   it was defined for the whole STIX-Package

        marking_dict = {}

        # We use queues rather than lists, since we have no need
        # to access elements somewhere in the list, but rather
        # always pop from the end

        import_first_queue = deque()

        pending_queue = deque()

        # The top-level object is certainly no marking, so we
        # put it on the pending queue

        pending_queue.append((top_level_id_and_rev_info, top_level_elt_name, top_level_elt_dict))

        while embedded_objects:
            # We go through the embedded objects and look for the 'import_first' label
            embedded_object = embedded_objects.pop()
            id_and_rev_info = embedded_object['id_and_rev_info']
            elt_name = embedded_object['elt_name']
            elt_dict = embedded_object['dict_repr']
            if 'import_first' in id_and_rev_info:
                import_first_queue.append((id_and_rev_info, elt_name, elt_dict))
            else:
                pending_queue.append((id_and_rev_info, elt_name, elt_dict))


        while import_first_queue:
            # We go through the import_first queue, import all markings,
            # and collect information about the STIX_Package in which the marking
            # was defined in the marking_dict (some organizations use a format
            # in which several STIX Packages are bundled into a single XML file).

            (id_and_rev_info, elt_name, elt_dict) = import_first_queue.pop()


            (info_obj, existed) = self.iobject_import(id_and_rev_info,
                                                      elt_name,
                                                      elt_dict)

            # id_and_rev_info can carries additional information other than the identifier
            # and timestamp that has been extracted: the MANTIS XML importer carries
            # along information inherited from ancestor objects: this allows us
            # to propagate information down to children, grandchildren etc. As we
            # see below, we use this mechanism to propagate information about
            # the STIX_Package in which an object was defined.

            if id_and_rev_info['inherited']['embedding_STIX_Package'] in marking_dict:
                marking_dict[id_and_rev_info['inherited']['embedding_STIX_Package']].append(info_obj)
            else:
                marking_dict[id_and_rev_info['inherited']['embedding_STIX_Package']] = [info_obj]


        while pending_queue:
            # Now we start the import of the remaining embedded objects (plus the top-level object)

            (id_and_rev_info, elt_name, elt_dict) = pending_queue.pop()

            if 'embedding_STIX_Package' in id_and_rev_info:
                embedding_STIX_Package = id_and_rev_info['embedding_STIX_Package']
            else:
                embedding_STIX_Package = id_and_rev_info.get('inherited',{}).get('embedding_STIX_Package')

            logger.debug("%s embedded in %s" % (id_and_rev_info['id'], embedding_STIX_Package))


            if elt_name in ['Kill_Chain','Kill_Chain_Phase']:
                # We chose not to attach markings to Kill_Chain information
                del(object_markings[:])
            else:
                # We mark the object with markings passed to the xml_import command
                # and markings defined in the STIX_Package in which the object was defined.

                object_markings = markings + marking_dict.get(embedding_STIX_Package,[])

            self.iobject_import(id_and_rev_info,
                                elt_name,
                                elt_dict,
                                markings=object_markings)

        # As we shall see below, we have configured the xml_importer such that
        # it recognizes OpenIOC structures embedded as test mechanism and
        # leaves them unprocessed. These unprocessed elements we now hand
        # over to the MANTIS OpenIOC importer.

        for unprocessed_elt in unprocessed_list:
            (id_and_rev_info,typeinfo,xml_node) = unprocessed_elt
            processor_class = self.processors.get(id_and_rev_info['defer_processing']['processor'],None)
            if processor_class:
                processor = processor_class(namespace_dict=self.namespace_dict)

                processor.xml_import(self,
                                     xml_content=xml_node,
                                     markings=markings,
                                     identifier_ns_uri=self.namespace_dict[id_and_rev_info['id'].split(':')[0]]
                )
            else:
                logger.error("Did not find a processor for %s" % id_and_rev_info['defer_processing']['processor'])


    # So, that was the main routine. Now we turn to the hooking functions used
    # for configuring the DINGOS XML importer.

    # The following regular expression is used to extract family (stix or cybox),
    # object type
    # from CybOX/STIX style namespace uris such as the following::
    #     http://cybox.mitre.org/objects#AddressObject-2
    #     http://cybox.mitre.org/common-2

    RE_LIST_NS_TYPE_FROM_NS_URL = [
        re.compile(
        "(?P<iotype_ns>http://(?P<family>(?P<family_tag>[^.]+)\.mitre.org)/([^#]+#)?(?P<type>.+?))((-|(_v))(?P<revision>.*))?$")]

    # In Cybox 1.x, the object properties were encompassed in an element called "Defined_Object". In the interest
    # of equal fact terms for equal things, we rename occurrences of "Defined_Object" to "Properties" upon
    # import. Thus, most fact terms will look the same for Cybox 1.x and Cybox 2.x

    RE_DEFINED_OBJECT = re.compile("Defined_Object")


    #
    # First of all, we define functions for the hooks provided to us
    # by the DINGO xml-import.
    #


    def id_and_revision_extractor(self, xml_elt):
        """
        Function for determing an identifier (and, where applicable, timestamp/revision
        information) for extracted embedded content;
        to be used for DINGO's xml-import hook 'id_and_revision_extractor'.

        This function is called

        - for the top-level node of the XML to be imported.

        - for each node at which an embedded object is extracted from the XML
          (when this occurs is governed by the following function, the
          embedding_pred

        It must return an identifier and, where applicable, a revision and or timestamp;
        in the form of a dictionary {'id':<identifier>, 'timestamp': <timestamp>}.
        How you format the identifier is up to you, because you will have to adopt
        the code in function xml_import such that the Information Objects
        are created with the proper identifier (consisting of qualifying namespace
        and uri.)

        Note: the xml_elt is an XMLNode defined by the Python libxml2 bindings. If you
        have never worked with these, have a look at

        - Mike Kneller's brief intro: http://mikekneller.com/kb/python/libxml2python/part1
        - the functions in django-dingos core.xml_utils module

        """
        result = {'id': None,
                  'timestamp': None,
        }


        if xml_elt.properties:
            attributes = extract_attributes(xml_elt, prefix_key_char='@')
            # Extract identifier:
            if '@id' in attributes:
                result['id'] = attributes['@id']
            elif '@object_reference' in attributes:
                # 'object_reference' is used as follows::
                #
                #     (...)
                #     <EmailMessageObj:Attachments>
                #           <EmailMessageObj:File xsi:type="FileObj:FileObjectType"
                #              object_reference="cybox:object-3cf6a958-5c3f-11e2-a06c-0050569761d3"/>
                #     </EmailMessageObj:Attachments>
                #     (...)
                result['id'] = attributes['@object_reference']

            elif '@phase_id' in attributes:
                result['id'] = attributes['@phase_id']
                result['extract_empty_embedded'] = True


            # Unfortunately, STIX/Cybox do not allow for addition of revision/timestamp information
            # along with the identifier of an object. If that does not change, we might start to use
            # a modified version of STIX/CybOX,
            # that allows 'revision_timestamp' information in an attribute whereever an 'id' or 'idref' attribute
            # is allowed.
            # We use a modified version of STIX/CybOX, in which each identifyable object can also carry
            # an attribute 'revision_timestamp'. We cannot use 'timestamp', because there
            # are CybOX elements such as 'Action' that carry a timestamp attribute with a different
            # semantic (the timestamp refers to the action, not the CybOX document)

            #if '@revision_timestamp' in attributes:
            #    result['timestamp'] = attributes['@revision_timestamp']

        if xml_elt.name == 'STIX_Package':
            result['embedding_STIX_Package'] = result['id']

        return result

    def stix_embedding_pred(self, parent, child, ns_mapping):
        """
        Predicate for recognizing inlined content in an XML; to
        be used for DINGO's xml-import hook 'embedded_predicate'.
        The question this predicate must answer is whether
        the child should be extracted into a separate object.

        The function returns either
        - False (the child is not to be extracted)
        - True (the child is extracted but nothing can be inferred
          about what kind of object is extracted)
        - a string giving some indication about the object type
          (if nothing else is known: the name of the element, often the
           namespace of the embedded object)
        - a dictionary, of the following form::

             {'id_and_revision_info' : { 'id': something/None,
                                         'ts': something/None,
                                          ... other information you want to
                                          record for this object for later usage,
                                       },
              'embedded_ns': False/True/some indication about object type as string}

        Note: the 'parent' and 'child' arguments are XMLNodes as defined
        by the Python libxml2 bindings. If you have never worked with these, have a look at

        - Mike Kneller's brief intro: http://mikekneller.com/kb/python/libxml2python/part1
        - the functions in django-dingos core.xml_utils module
        """

        def extract_typeinfo(child):
            """
            When recognizing an embedding object, we try to find out what
            kind of object it is
            """

            # Let's try to find a grandchild and return the namespace of this grandchild:
            # This can be used as indicator for the object type that is referenced here.

            # Let's further try to find a grandchild (if there is one)

            grandchild = child.children
            type_info = None

            while grandchild is not None:
                try:
                    grandchild_attrs = extract_attributes(grandchild, prefix_key_char='')
                    if 'xsi:type' in grandchild_attrs and grandchild.name=='Properties':
                        type_info = grandchild_attrs['xsi:type'].split(':')[0]
                    else:
                        type_info = grandchild.ns().name
                    break

                except:
                    # This catches if the grandchild does not have a namespace
                    grandchild = grandchild.next
            if type_info:
                logger.debug("Found type info %s" % type_info)
                return type_info
            else:
                logger.debug("Embedding, but did not find type info")
                return True

        child_attributes = extract_attributes(child, prefix_key_char='')

        parent_attrs = extract_attributes(parent, prefix_key_char='')


        # We start with some special cases. If we find
        # OpenIOC content in a test mechanism, we use the embedded-predicate
        # to pass along additional information: by setting the key 'defer_processing'
        # in the 'id_and_revision_info', we tell the DINGOS xml_importer to
        # return the child element unprocessed.

        if parent.name=='Test_Mechanism':
            if 'xsi:type' in parent_attrs:
                if 'OpenIOC2010TestMechanismType' in parent_attrs['xsi:type']:
                    # We have an embedded OpenIOC document.

                    # We extract
                    id_and_revision_info = OpenIOC_Importer.id_and_revision_extractor(child)
                    id_and_revision_info['defer_processing'] = {'processor': 'OpenIOC2010'}
                    return {'embedded_ns':child.ns().name,
                            'id_and_revision_info':id_and_revision_info}



        if parent.name=='Kill_Chain' and child.name=='Kill_Chain_Phase':
            # We also extract kill-chain phases. Not sure whether that
            # is the best idea, though.
            logger.debug("Found killchain phase %s" % extract_typeinfo(child))
            return extract_typeinfo(child)


        if parent.name=='Handling' and child.name=='Marking' and parent.get_parent().name =='STIX_Header':

            # We also cut out Markings, because we need an InfoObject with the Marking
            # contents in order to create markings for the import.
            # We seed the id_and_revision info for this object with a tag 'import_first',
            # which will be read later below in the xml_import function: marking objects
            # will be imported first and a marking dictionary will provide information on
            # which STIX Package provided which marking.

            # Markings do not carry identifiers. The xml_importer will automagically create
            # an identifier derived from the identifier of the encompassing object.

            return {'embedded_ns': extract_typeinfo(child),
                    'id_and_revision_info' : {'import_first': True}}

        # Finally, the standard case: an 'id' attribute signifies
        # an embedded object that we want to extract.
        if ('id' in child_attributes):
            return extract_typeinfo(child)

        # 'object_reference' is used in Cybox as follows::
        #
        #     (...)
        #     <EmailMessageObj:Attachments>
        #           <EmailMessageObj:File xsi:type="FileObj:FileObjectType"
        #              object_reference="cybox:object-3cf6a958-5c3f-11e2-a06c-0050569761d3"/>
        #     </EmailMessageObj:Attachments>
        #     (...)
        #
        # That is actually a reference, but we need to turn it into an '@idref'-reference.
        # By treating the 'File' object-reference as an embedded object, this is done
        # automatically, because the xml_importer replaces embedded content with
        # '@idref'-based references.

        if 'object_reference' in child_attributes:
            return extract_typeinfo(child)

        if child.name=='Object' and not 'idref' in child_attributes:
            # Unfortunately, e.g., the example files created by MITRE from Mandiant reports
            # and OpenIOCs give an identifier to an observable, but not to the
            # object embedded in the observable. We, however, need an identifier for
            # the object, because otherwise the whole machinery that infers an object's
            # type does not work. So, if we find an object without identifier that
            # is embedded in an observable with identifier, we also want to extract
            # the object ... and need to derive the object identifier from the
            # observable identifier. This is done automagically by the xml_import
            # function itself.

            return extract_typeinfo(child)
        return False


    # Next, we define functions for the hooks provided by the
    # 'from_dict' method of DINGO InfoObject Objects, through
    # which the shape of the InfoObjects generated from the
    # dictionary representation created by the xml importer
    # is influenced.
    #
    # The hooking is carried out by defining a list
    # containing pairs of predicates (i.e., a function returning True or False)
    # and an associated hooking function. For each InfoObject2Fact object
    # the list is iterated by applying the predicate to input data.
    # If the predicate returns True, then the hooking function is applied
    # and may change the parameters for creation of fact.

    # What is usually at least required here is a 'reference handler' that
    # knows how to deal with references created by the import when extracting
    # an embedded object.

    #    The signature of a predicate is as follows:
    #
    # - Inputs:
    #   - fact dictionary of the following form::
    #
    #        { 'node_id': 'N001:L000:N000:A000',
    #          'term': 'Hashes/Hash/Simple_Hash_Value',
    #          'attribute': 'condition' / False,
    #          'value': u'Equals'
    #        }
    #   - attr_info:
    #     A dictionary with mapping of XML attributes concerning the node in question
    #     (note that the keys do *not* have a leading '@' unless it is an internally
    #     generated attribute by Dingo.
    #
    # - Output: Based on these inputs, the predicate must return True or False. If True
    #   is returned, the associated handler function is run.
    #
    # The signature of a handler function is as follows:
    #
    # - Inputs:
    #
    #   - info_object: the information object to which the fact is to be added
    #   - fact: the fact dictionary of the following form::
    #        { 'node_id': 'N001:L000:N000:A000',
    #          'term': 'Hashes/Hash/Simple_Hash_Value',
    #          'attribute': 'condition' / False,
    #          'value': u'Equals'
    #        }
    #   - attr_info:
    #     A dictionary with mapping of XML attributes concerning the node in question
    #     (note that the keys do *not* have a leading '@' unless it is an internally
    #     generated attribute by Dingo.
    #
    #   - add_fact_kargs:
    #     The arguments with which the fact will be generated after all handler functions
    #     have been called. The dictionary contains the following keys::
    #
    #         'fact_dt_kind' : <FactDataType.NO_VOCAB/VOCAB_SINGLE/...>
    #         'fact_dt_namespace_name': <human-readable shortname for namespace uri>
    #         'fact_dt_namespace_uri': <namespace uri for datataype namespace>
    #         'fact_term_name' : <Fact Term such as 'Header/Subject/Address'>
    #         'fact_term_attribute': <Attribute key such as 'category' for fact terms describing an attribute>
    #         'values' : <list of FactValue objects that are the values of the fact to be generated>
    #         'node_id_name' : <node identifier such as 'N000:N000:A000'
    #
    # - Outputs:
    #
    #   The handler function outputs either True or False: If False is returned,
    #   then the fact will *not* be generated. Please be aware that if you use this option,
    #   then there will be 'missing' numbers in the sequence of node ids.
    #   Thus, if you want to suppress the creation of facts for attributes,
    #   rather use the hooking function 'attr_ignore_predicate'
    #
    #   As side effect, the function can make changes to the dictionary passed in parameter
    #   'add_fact_kargs' and thus change the fact that will be created.




    def cybox_RAW_ft_handler(self, enrichment, fact, attr_info, add_fact_kargs):
        """
        Handler for facts whose content is to be written to disk rather
        than stored in the database. We use it for all elements
        that contain the string 'Raw_' ('Raw_Header', 'Raw_Artifact', ...)
        """
        # get the value
        raw_value = add_fact_kargs['values'][0]

        if len(raw_value) >= RAW_DATA_TO_DB_FOR_LENGTH_LESS_THAN:
            # rewrite the argument for the creation of the fact: there are
            # no values to be added to the database
            (value_hash,storage_location) = write_large_value(raw_value,
                                                              dingos.DINGOS_LARGE_VALUE_DESTINATION)

            add_fact_kargs['values'] = [(value_hash,storage_location)]


        return True


    def reference_handler(self, iobject, fact, attr_info, add_fact_kargs):
        """
        Handler for facts that contain a reference to a fact.

        As shown below in the handler list, this handler is called
        when a attribute with key '@idref' on the fact's node
        is detected -- this attribute signifies that this fact does not contain
        a value but points to another object. Thus we either retrieve
        the object or, if an object with the given id does not yet exist,
        create a PLACEHOLDER object.

        We further create/refer to the fitting fact data type:
        we want the fact data type to express that the fact is
        a reference to an object.
        """

        if 'idref' in attr_info:
            ref_key = 'idref'
            (namespace, namespace_uri, uid) = self.split_qname(attr_info[ref_key])
        elif fact['attribute'] and fact['attribute'] == 'phase_id':

            if fact['term']=='':
                return True
            else:
                (namespace, namespace_uri, uid) = self.split_qname(fact['value'])
        elif fact['attribute'] and fact['attribute'] == 'kill_chain_id':
            if fact['term'] == '':
                return True
            else:
                (namespace, namespace_uri, uid) = self.split_qname(fact['value'])




        timestamp = None

        #if '@revision_timestamp' in attr_info:
        #    timestamp = attr_info['@revision_timestamp']

        if not timestamp:
            timestamp = self.create_timestamp

        (target_mantis_obj, existed) = MantisImporter.create_iobject(
            uid=uid,
            identifier_ns_uri=namespace_uri,
            timestamp=timestamp)

        logger.debug("Creation of Placeholder for %s %s returned %s" % (namespace_uri, uid, existed))
        add_fact_kargs['value_iobject_id'] = Identifier.objects.get(uid=uid, namespace__uri=namespace_uri)

        return True


    def cybox_valueset_fact_handler(self, enrichment, fact, attr_info, add_fact_kargs):
        """
        Handler for dealing with 'value_set' values.

        Unfortunately, CybOX et al. sometimes use comma-separated
        value lists rather than an XML structure that can contain
        several values.

        This handler is called for elements concerning a value-set
        such as the following example::

            <URIObj:Value condition="IsInSet"
            value_set="www.sample1.com/index.html, sample2.com/login.html, dev.sample3.com/index/kb.html"
            datatype="AnyURI"/>

        """

        value_list = attr_info['value_set'][fact['node_id']].split(",")
        value_list = map(lambda x: x.strip(), value_list)

        add_fact_kargs['values'] = value_list

        return True


    def cybox_csv_handler(self, enrichment, fact, attr_info, add_fact_kargs):
        """
        Handler for dealing with comma-separated values.


        Unfortunately, CybOX et al. sometimes use comma-separated
        value lists. Or rather, since Cybox 2.0.1, '##comma##'-separated lists.

        At least now we can easily recognize comma-separated lists ...
        until one joker starts putting the string '##comma##' into
        his malware and email, that is...

        The handler itself is rather straightforward.
        """

        # Since Cybox 2.0.1, '##comma##' is to be used instead of ','. So,
        # we first check whether '##comma##' occurs -- if so, we take
        # that as separator; if not, we take ','

        if '##comma##' in fact['value']:
            separator = '##comma##'
        else:
            separator = ','

        value_list = map(lambda x: x.strip(), fact['value'].split(separator))

        add_fact_kargs['values'] = value_list

        return True

    def cybox_csv_predicate(self, fact, attr_info):
        """
        Predicate for dealing with comma-separated values.

        Unfortunately, CybOX et al. sometimes use comma-separated
        value lists. Or rather, since Cybox 2.0.1, '##comma##'-separated lists.

        At least now we can easily recognize comma-separated lists ...
        until one joker starts putting the string '##comma##' into
        his malware and email, that is...
        """

        #
        return ('##comma##' in fact['value'])



    def cybox_defined_object_in_fact_term_handler(self, enrichment, fact, attr_info, add_fact_kargs):
        """
        From CybOX 1.x to Cybox 2.0.0, there was a structural change in the way
        observable properties are included in the XML: in CybOX 1.x, they
        were embedded in an element called 'Defined_Object' -- since CybOX 2.x,
        we have the 'Properties' element. Here, we rename occurrences of 'Defined_Object'
        in a fact term with 'Properties'. As a result, fact terms for, e.g.,
        'Properties/Header/To/Recipient/AddressValue' for an Email object are the same
        for imports from Cybox 1.x and Cybox 2.x.
        """
        add_fact_kargs['fact_term_name'] = self.RE_DEFINED_OBJECT.sub('Properties', fact['term'])
        return True


    def cybox_defined_object_in_fact_term_predicate(self, fact, attr_info):
        """
        From CybOX 1.x to Cybox 2.0.0, there was a structural change in the way
        observable properties are included in the XML: in CybOX 1.x, they
        were embedded in an element called 'Defined_Object' -- since CybOX 2.x,
        we have the 'Properties' element. Here, we rename occurrences of 'Defined_Object'
        in a fact term with 'Properties'. As a result, fact terms for, e.g.,
        'Properties/Header/To/Recipient/AddressValue' for an Email object are the same
        for imports from Cybox 1.x and Cybox 2.x.
        """
        return self.RE_DEFINED_OBJECT.search(fact['term'])

    def force_nonleaf_fact_predicate(self, fact_dict, attr_dict):

        """
        The force_noneleaf_fact_predicate determines whether a fact should
        be created for a non-leaf XML node. Consider the following structure::

            <cybox:Related_Objects>
                <cybox:Related_Object idref="example:URI-f9daa775-ad68-4a94-b110-549e7f26bf19">
                    <cybox:Relationship xsi:type="cyboxVocabs:ObjectRelationshipVocab-1.0">Connected_To</cybox:Relationship>
                </cybox:Related_Object>
            </cybox:Related_Objects>

        BY default, the importer would not create a fact with fact term
        "Related_Objects/Related_Object", because this does not denote a leaf
        (either a value or an attribute) in the XML structure. But in this particular
        case, we need the fact "Related_Objects/Related_Object", because this is
        where the reference to the referenced object will be attached.

        This is what the force_nonleaf_predicate is for: it takes two arguments:
        firstly, a fact dictionary of the following form::

               { 'node_id': 'N001:L000:',
                 'term': 'Related_Objects/Related_Object',
                 'attribute': 'condition',
                 'value': u''
               }

        Secondly, a dictionary with key-value pairs of all attributes (identitified
        by a leading '@' attached directly to the node.

        If the predicate returns 'True', then a fact is created for a non-leaf
        element; otherwise, no fact is created.

        """

        return 'Related_Object' in fact_dict['term'] and '@idref' in attr_dict

    def attr_ignore_predicate(self, fact_dict):
        """
        The attr_ignore predicate is called for each fact that would be generated
        for an XML attribute. It takes a fact dictionary of the following form
        as input::
               { 'node_id': 'N001:L000:N000:A000',
                 'term': 'Hashes/Hash/Simple_Hash_Value',
                 'attribute': 'condition',
                 'value': u'Equals'
               }

        If the predicate returns 'False, the fact is *not* created. Note that, nevertheless,
        during import, the information about this attribute is available to
        the attributed fact as part of the 'attr_dict' that is generated for the creation
        of each fact and passed to the handler functions called for the fact.

        """
        if not fact_dict['attribute']:
            return False

        if '@' in fact_dict['attribute']: # and not fact_dict['attribute'] == '@ns':
            # We remove all attributes added by Dingo during import

            #if (fact_dict['attribute']=='@ns'
            #    and
            #        (('extension' in fact_dict['value']
            #         or
            #         ((not 'stix' in fact_dict['value'])
            #         and
            #         (not 'cybox' in fact_dict['value'])
            #         )
            #         )
            #        )
            #):
            #    return False
            #else:
            #    return True

            return True

        attr_key = fact_dict['attribute']

        cybox_attr_ignore_list = [# we drop id-attributes:
                                  # everything that has an identifier gives rise to a new object and
                                  # the identifier is used then and there,
                                  'id' ,
                                  'object_reference',
                                  'idref',
                                  # Type information we have already read and treated,
                                  # so no need to keep it around
                                  'xsi:type',
                                  'datatype',
                                  'type',
                                  # value_set attributes are treated by a special handler
                                  'value_set',
                                  # no need to retain the schemaLocation info on top-level.
                                  'xsi:schemaLocation'
        ]

        if attr_key in cybox_attr_ignore_list:
            return True
        return False


    def fact_handler_list(self):
        return [
            # We write the content of elements with "Raw" in the fact term to disk
            # rather than storing them in the database.
            (lambda fact, attr_info: (not fact['attribute']) and "Raw_" in fact['term'],
             self.cybox_RAW_ft_handler),
            # When we find a reference, we either retrieve the referenced object from
            # the database (if it exists) or we generate a PLACEHOLDER object
            (lambda fact, attr_info:    ("idref" in attr_info)
            or (fact['attribute']
                and ("phase_id" in fact['attribute']
                     or ("kill_chain_id" in fact['attribute']))),
             self.reference_handler),
            # We normalize elements containing comma-separated values
            (self.cybox_csv_predicate, self.cybox_csv_handler),
            # We also normalize comma-separated values in value-set attributes
            (lambda fact, attr_info: 'value_set' in attr_info, self.cybox_valueset_fact_handler),
            # We rename 'Defined_Object' in fact terms from Cybox 1.0 to 'Properties' of Cybox 2.0
            (self.cybox_defined_object_in_fact_term_predicate, self.cybox_defined_object_in_fact_term_handler),

        ]


    #
    # Finally: we define the function that takes a fact with associated
    # attribute information and determines the fact data type for this
    # fact.
    #

    def cybox_datatype_extractor(self, enrichment, fact, attr_info, namespace_mapping, add_fact_kargs):
        """

        The datatype extractor is called for each fact with the aim of determining the fact's datatype.
        The extractor function has the following signature:

        - Inputs:
          - info_object: the information object to which the fact is to be added
          - fact: the fact dictionary of the following form::
               { 'node_id': 'N001:L000:N000:A000',
                 'term': 'Hashes/Hash/Simple_Hash_Value',
                 'attribute': 'condition' / False,
                 'value': u'Equals'
               }
          - attr_info:
            A dictionary with mapping of XML attributes concerning the node in question
            (note that the keys do *not* have a leading '@' unless it is an internally
            generated attribute by Dingo.
          - namespace_mapping:
            A dictionary containing the namespace mapping extracted from the imported XML file.
          - add_fact_kargs:
            The arguments with which the fact will be generated after all handler functions
            have been called. The dictionary contains the following keys::

                'fact_dt_kind' : <FactDataType.NO_VOCAB/VOCAB_SINGLE/...>
                'fact_dt_namespace_name': <human-readable shortname for namespace uri>
                'fact_dt_namespace_uri': <namespace uri for datataype namespace>
                'fact_term_name' : <Fact Term such as 'Header/Subject/Address'>
                'fact_term_attribute': <Attribute key such as 'category' for fact terms describing an attribute>
                'values' : <list of FactValue objects that are the values of the fact to be generated>
                'node_id_name' : <node identifier such as 'N000:N000:A000'

        Just as the fact handler functions, the datatype extractor can change the add_fact_kargs dictionary
        and thus change the way in which the fact is created -- usually, this ability is used to change
        the following items in the dictionary:

        - fact_dt_name
        - fact_dt_namespace_uri
        - fact_dt_namespace_name (optional -- the defining part is the uri)
        - fact_dt_kind

        The extractor returns "True" if datatype info was found; otherwise, False is returned
        """

        if "idref" in attr_info:
            # Set up the fact data type as a data type that expresses
            # the referencing that is going on
            embedded_type_info = attr_info.get('@embedded_type_info', None)
            logger.debug("Embedded type info %s" % embedded_type_info)
            type_info = self.derive_iobject_type(attr_info["@ns"],
                                                 embedded_type_info,
                                                 fact['term'].split('/')[-1])

            add_fact_kargs['fact_dt_name'] = type_info['iobject_type_name']
            add_fact_kargs['fact_dt_namespace_uri'] = type_info['iobject_type_namespace_uri']
            add_fact_kargs['fact_dt_kind'] = FactDataType.REFERENCE

            return True

        if "xsi:type" in attr_info:
            # Use xsi:type attribute
            add_fact_kargs['fact_dt_name'] = attr_info["xsi:type"]

        elif "datatype" in attr_info and fact['node_id'] in attr_info["datatype"]:
            # Use 'datatype' attribute
            add_fact_kargs['fact_dt_name'] = attr_info["datatype"]
        else:
            return False

        # It may be the case that the datatype information contains namespace info
        if ':' in add_fact_kargs['fact_dt_name']:
            add_fact_kargs['fact_dt_kind'] = FactDataType.VOCAB_SINGLE
            add_fact_kargs['fact_dt_namespace_name'] = add_fact_kargs['fact_dt_name'].split(':')[0]
            add_fact_kargs['fact_dt_name'] = add_fact_kargs['fact_dt_name'].split(':')[1]

            add_fact_kargs['fact_dt_namespace_uri'] = namespace_mapping.get(add_fact_kargs['fact_dt_namespace_name'],
                                                                            '%s/%s' % (DINGOS_NAMESPACE_URI, (
                                                                                enrichment.iobject_family.name)))

        return True


    def split_qname(self, cybox_id):
        """
        Separate the namespace from the identifier in a qualified name and lookup the namespace URI associated
        with the given namespace.
        """
        if ':' in cybox_id:
            (namespace, uid) = cybox_id.split(':', 1)
        else:
            namespace = None
            uid = cybox_id

        if namespace and namespace in self.namespace_dict:
            namespace_uri = self.namespace_dict[namespace]
        else:
            logger.warning("Could not retrieve namespace for identifier %s" % (cybox_id))
            # TODO: Introduce configurable URI
            namespace_uri = None

        if not namespace_uri:
            if self.default_identifier_ns_uri:
                namespace_uri = self.default_identifier_ns_uri
            else:
                namespace_uri = "%s/%s" % (DINGOS_MISSING_ID_NAMESPACE_URI_PREFIX, namespace)

        return (namespace, namespace_uri, uid)

    def derive_iobject_type(self, embedding_ns, embedded_ns, elt_name):
        """
        Derive type of information object stemming from an embedded element
        based on namespace information of embedding element, the embedded
        element itself, and the name of the element.

        """


        # Extract namespace-information

        ns_info = search_by_re_list(self.RE_LIST_NS_TYPE_FROM_NS_URL, self.namespace_dict.get(embedding_ns, ""))

        if not ns_info:
            ns_info = {}

        # This should yield the following information:
        # - For namespace of an Cybox Object such as http://cybox.mitre.org/objects#AddressObject-2:
        #   - iotype_ns = http://cybox.mitre.org/objects#AddressObject
        #   - family = cybox.mitre.org
        #   - family_tag = cybox
        #   - type = AddressObject
        #   - revision = 2
        # - For a base namespace such as http://cybox.mitre.org/common-2:
        #   - iotype_ns = http://cybox.mitre.org/common
        #   - family = cybox.mitre.org
        #   - family_tag = cybox
        #   - type = common
        #   - revision = 2

        iobject_family_name = ns_info.get('family',None)
        if not iobject_family_name:
            iobject_family_name = ""
        family_info = {}

        if ns_info.get('family_tag',None) in ['stix', 'cybox']:
            family_info = search_by_re_list(self.RE_LIST_NS_TYPE_FROM_NS_URL,
                                            self.namespace_dict.get(ns_info['family_tag'], ""))
            iobject_family_revision_name = family_info["revision"]

        else:
            iobject_family_revision_name = ns_info.get("revision",None)
        if not iobject_family_revision_name:
            iobject_family_revision_name = ''

        # We take the object type from the ``xsi:type`` attribute
        # given as in the following example::
        #    <cybox:Properties xsi:type="EmailMessageObj:EmailMessageObjectType">
        #    <cybox:Properties xsi:type="AddrObj:AddressObjectType" category="ipv4-addr">
        #
        if embedded_ns:
            namespace_uri = self.namespace_dict.get(embedded_ns, "")
            type_info = search_by_re_list(self.RE_LIST_NS_TYPE_FROM_NS_URL, namespace_uri)
            if not type_info:
                type_info = {}
            if type_info and type_info.get('type',None) in ['common', 'cybox', 'stix']:
                iobject_type_name = elt_name
                iobject_type_namespace_uri = ns_info['iotype_ns']
                iobject_type_revision_name = ns_info['revision']
            else:
                iobject_type_namespace_uri = type_info.get('iotype_ns',"%s/%s" % (dingos.DINGOS_MISSING_ID_NAMESPACE_URI_PREFIX,embedded_ns))
                iobject_type_name = type_info.get('type',embedded_ns)#.split('Object')[0]
                iobject_type_revision_name = type_info.get('revision','')

        else:
            iobject_type_name = elt_name
            iobject_type_revision_name = iobject_family_revision_name
            iobject_type_namespace_uri = ns_info.get("iotype_ns", "%s/%s" % (dingos.DINGOS_MISSING_ID_NAMESPACE_URI_PREFIX,elt_name))


        if not iobject_type_revision_name:
            iobject_type_revision_name = ''

        logger.debug("Results of datatype extraction for ns %s, embedded ns %s and element name %s" % (
        embedding_ns, embedded_ns, elt_name))
        logger.debug("Family Name: %s" % iobject_family_name)
        logger.debug("Family Revision %s" % iobject_family_revision_name)
        logger.debug("Type Name %s" % iobject_type_name)
        logger.debug("Type NS URI %s" % iobject_type_namespace_uri)
        logger.debug("Type Revision %s" % iobject_type_revision_name)

        return {'iobject_type_name': iobject_type_name,
                'iobject_type_revision_name': iobject_type_revision_name,
                'iobject_type_namespace_uri': iobject_type_namespace_uri,
                'iobject_family_name': iobject_family_name,
                'iobject_family_revision_name': iobject_family_revision_name}




    def iobject_import(self,
                       id_and_rev_info,
                       elt_name,
                       obj_dict,
                       markings=None,
                       cybox_id=None):
        """
        Derives InfoObjectType and import InfoObjectType

        """

        iobject_type_ns = None

        # Derive the namespace information
        if ('@xsi:type' in obj_dict or
                    '@@embedded_type_info' in obj_dict or
                    '@xsi:type' in obj_dict.get('Properties', {}) or
                    '@xsi:type' in obj_dict.get('Defined_Object', {}) ):
            if '@xsi:type' in obj_dict:
                iobject_type_ns = obj_dict['@xsi:type'].split(':')[0]
            elif '@xsi:type' in obj_dict.get('Properties', {}):
                iobject_type_ns = obj_dict['Properties']['@xsi:type'].split(':')[0]
            elif '@xsi:type' in obj_dict.get('Defined_Object', {}):
                iobject_type_ns = obj_dict['Defined_Object']['@xsi:type'].split(':')[0]
            else:
                iobject_type_ns = obj_dict['@@embedded_type_info']

        # Find out what the type of the Information Object to be created should be
        type_info = self.derive_iobject_type(obj_dict['@@ns'], iobject_type_ns, elt_name)

        if not 'id' in id_and_rev_info or not id_and_rev_info['id']:
            logger.info("Object of type %s without id information encountered, skipping" % elt_name)
            return
        (namespace, namespace_uri, uid) = self.split_qname(id_and_rev_info['id'])

        (info_obj, existed) = MantisImporter.create_iobject(iobject_family_name=type_info['iobject_family_name'],
                                                            iobject_family_revision_name=type_info[
                                                                'iobject_family_revision_name'],
                                                            iobject_type_name=type_info['iobject_type_name'],
                                                            iobject_type_namespace_uri=type_info[
                                                                'iobject_type_namespace_uri'],
                                                            iobject_type_revision_name=type_info[
                                                                'iobject_type_revision_name'],
                                                            iobject_data=obj_dict,
                                                            uid=uid,
                                                            identifier_ns_uri=namespace_uri,
                                                            timestamp=self.create_timestamp,
                                                            create_timestamp=self.create_timestamp,
                                                            markings=markings,
                                                            config_hooks={
                                                            'special_ft_handler': self.fact_handler_list(),
                                                            'datatype_extractor': self.cybox_datatype_extractor,
                                                            'attr_ignore_predicate': self.attr_ignore_predicate,
                                                            'force_nonleaf_fact_predicate': self.force_nonleaf_fact_predicate},
                                                            namespace_dict=self.namespace_dict,
        )

        return (info_obj, existed)








