#!/usr/bin/env python

from __future__ import print_function

import csv

from django.core.management.base import LabelCommand

from mapit.models import Area


class Command(LabelCommand):
    help = 'Analyse a CVS file generated by mapit_global_find_differences'
    args = '<CSV-FILE>'

    def handle_label(self, differences_results_csv_file, **options):

        osm_elements_seen_in_new_data = set([])

        with open(differences_results_csv_file) as fp:
            reader = csv.DictReader(fp)
            equals_but_not_equals_exact = 0
            equals_exact_but_not_equals = 0
            completely_new_boundaries = 0
            boundary_stayed_the_same = 0
            boundary_changed = 0
            empty_in_either = 0
            total = 0
            for row in reader:
                total += 1
                counted_somewhere = False
                if row['GEOSEquals'] == 'True' and row['GEOSEqualsExact'] == 'False':
                    equals_but_not_equals_exact += 1
                    counted_somewhere = True
                elif row['GEOSEqualsExact'] == 'True' and row['GEOSEquals'] == 'False':
                    equals_exact_but_not_equals += 1
                    counted_somewhere = True
                if row['ExistedPreviously'] == 'False':
                    # n.b. includes cases where the new area is empty or malformed
                    completely_new_boundaries += 1
                    counted_somewhere = True
                elif row['GEOSEquals'] == 'True':
                    boundary_stayed_the_same += 1
                    counted_somewhere = True
                elif row['GEOSEquals'] == 'False':
                    boundary_changed += 1
                    counted_somewhere = True
                elif row['PreviousEmpty'] == 'True' or row['NewEmpty'] == 'True':
                    empty_in_either += 1
                    counted_somewhere = True
                if not counted_somewhere:
                    print("not counted:", row)
                osm_elements_seen_in_new_data.add((row['ElementType'], row['ElementID']))

        disappeared = 0

        for a in Area.objects.all().iterator():
            all_codes = a.codes.all()
            if len(all_codes) == 0:
                print("code missing for:", a)
            elif len(all_codes) > 1:
                print("too many codes (%d) for: %s" % (len(all_codes), a))
            else:
                code = all_codes[0]
                element_type = 'relation' if code.type.code == 'osm_rel' else 'way'
                element_id = code.code
                t = (element_type, element_id)
                if t not in osm_elements_seen_in_new_data:
                    disappeared += 1
                    polygons = a.polygons.all()
                    if len(polygons) > 0:
                        if polygons[0].polygon.valid:
                            lon, lat = polygons[0].polygon.point_on_surface
                            point_url = "/point/4326/%s,%s.html" % (lon, lat)
                        else:
                            point_url = "[first polygon was invalid, skipping]"
                    else:
                        point_url = "[no polygons]"
                    print("disappeared: /code/%s/%s - a point inside is: %s" % (code.type.code, code.code, point_url))

        print("========================================================================")
        print("equals_but_not_equals_exact:", equals_but_not_equals_exact)
        print("equals_exact_but_not_equals:", equals_exact_but_not_equals)
        print("completely_new_boundaries:", completely_new_boundaries)
        print("areas that disappeared:", disappeared)
        print("boundary_stayed_the_same:", boundary_stayed_the_same)
        print("boundary_changed:", boundary_changed)
        print("empty_in_either:", empty_in_either)
        print("out of a total:", total)
