from django.db import models
from django.utils.encoding import smart_unicode
from exceptions import Exception
from pymonon import Money, Currency, CURRENCIES
from monon import forms

from decimal import Decimal

__all__ = ('MoneyField', 'currency_field_name', 'NotSupportedLookup')

currency_field_name = lambda name: "%s_currency" % name
SUPPORTED_LOOKUPS = ('exact', 'lt', 'gt', 'lte', 'gte')


class NotSupportedLookup(Exception):
    def __init__(self, lookup):
        self.lookup = lookup

    def __str__(self):
        return "Lookup '%s' is not supported for MoneyField" % self.lookup


class MoneyFieldProxy(object):
    def __init__(self, field):
        self.field = field
        self.currency_field_name = currency_field_name(self.field.name)

    def _money_from_obj(self, obj):
        return Money(obj.__dict__[self.field.name], obj.__dict__[self.currency_field_name])

    def __get__(self, obj, type=None):
        if obj is None:
            raise AttributeError('Can only be accessed via an instance.')
        if not isinstance(obj.__dict__[self.field.name], Money):
            obj.__dict__[self.field.name] = self._money_from_obj(obj)
        return obj.__dict__[self.field.name]

    def __set__(self, obj, value):
        if isinstance(value, Money):
            obj.__dict__[self.field.name] = value.amount
            setattr(obj, self.currency_field_name, value.currency.code)
        else:
            if value:
                value = str(value)
            obj.__dict__[self.field.name] = self.field.to_python(value)


class CurrencyField(models.CharField):
    def __init__(self, verbose_name=None, name=None, default=Currency.get_default(), **kwargs):
        if isinstance(default, Currency):
            default = default.code
        kwargs['max_length'] = 3
        super(CurrencyField, self).__init__(verbose_name, name, default=default, **kwargs)

    def get_internal_type(self):
        return "CharField"

    def get_prep_value(self, value):
        if isinstance(value, Currency):
            value = value.code
        else:
            value = Currency(value).code
        return value


class MoneyField(models.DecimalField):
    def __init__(self, verbose_name=None, name=None,
                 max_digits=13, decimal_places=3,
                 default=Decimal("0.0"), default_currency=Currency.get_default(), **kwargs):
        if isinstance(default, Money):
            self.default_currency = default.currency

        # Avoid giving the user hard-to-debug errors if they miss required attributes
        if max_digits is None:
            raise Exception("You have to provide a max_digits attribute to Money fields.")

        if decimal_places is None:
            raise Exception("You have to provide a decimal_places attribute to Money fields.")

        self.default_currency = default_currency
        super(MoneyField, self).__init__(verbose_name, name, max_digits, decimal_places, default=default, **kwargs)

    def to_python(self, value):
        if isinstance(value, Money):
            value = value.amount
        return super(MoneyField, self).to_python(value)

    def get_internal_type(self):
        return "DecimalField"

    def contribute_to_class(self, cls, name):
        c_field_name = currency_field_name(name)
        c_field = CurrencyField(max_length=3, default=self.default_currency, editable=False)
        c_field.creation_counter = self.creation_counter
        cls.add_to_class(c_field_name, c_field)

        super(MoneyField, self).contribute_to_class(cls, name)

        setattr(cls, self.name, MoneyFieldProxy(self))

        from managers import money_manager

        if getattr(cls, '_default_manager', None):
            cls._default_manager = money_manager(cls._default_manager)
        elif hasattr(cls, 'objects'):
            cls.objects = money_manager(cls.objects)
        else:
            cls.objects = money_manager(models.Manager)

    def get_db_prep_save(self, value, connection):
        if isinstance(value, Money):
            value = value.amount
        return super(MoneyField, self).get_db_prep_save(value, connection)

    def get_db_prep_lookup(self, lookup_type, value, connection, prepared=False):
        if not lookup_type in SUPPORTED_LOOKUPS:
            raise NotSupportedLookup(lookup_type)
        value = self.get_db_prep_save(value, connection)
        return super(MoneyField, self).get_db_prep_lookup(lookup_type, value, connection, prepared)

    def get_default(self):
        if isinstance(self.default, Money):
            return self.default
        else:
            return super(MoneyField, self).get_default()

    def formfield(self, **kwargs):
        defaults = {'form_class': forms.MoneyField}
        defaults.update(kwargs)
        return super(MoneyField, self).formfield(**defaults)

## South support
try:
    from south.modelsinspector import add_introspection_rules

    rules = [
        ((MoneyField,),
         [],  # No positional args
         {'default_currency':('default_currency', {})}),
        ((CurrencyField,),
         [],  # No positional args
         {}),  # No new keyword args
    ]

    add_introspection_rules(rules, ["^monon\.models"])
except ImportError:
    pass
