import abc
from collections import OrderedDict
from django.utils.http import urlencode
from six import string_types

from django.core.urlresolvers import reverse, NoReverseMatch
from django.utils.functional import cached_property

from navigator import settings


class Node(object):
    """
    Node object
    """

    def __init__(self, title, path, url_name=None, url_params=None, permissions=None,
                 populate_children=True, children=None, is_global=False, **kwargs):
        self.title = title
        self.path = path
        self.url_name = url_name
        if url_params is None:
            url_params = {}
        self.url_params = url_params
        if permissions is None:
            permissions = []
        self.permissions = permissions
        if children is None:
            children = {}
        self.populate_children = populate_children
        self.children = children
        self.is_global = is_global
        self.kwargs = kwargs

    @cached_property
    def url(self):
        query_string = urlencode(self.url_params)
        query = '?{}'.format(query_string) if query_string else ''
        try:
            if isinstance(self.url_name, tuple):
                return '{0}{1}'.format(
                    reverse(self.url_name[0], **self.url_name[1]), query)
            elif isinstance(self.url_name, string_types):
                return '{0}{1}'.format(reverse(self.url_name), query)
        except NoReverseMatch:
            pass
        return None

    def populate(self, request, parent_path=None, alias=None):

        if parent_path and not self.is_global:
            if self.path:
                active_path = '{}.{}'.format(parent_path, self.path)
            else:
                active_path = parent_path
        else:
            active_path = self.path if self.path else ''

        check_path = '{}.{}'.format(alias, active_path) if alias else active_path

        request_paths = getattr(request, settings.NAVIGATOR_PATH_ATTR, [])
        active = any([path.startswith(check_path) for path in request_paths])

        if self.children:
            children = OrderedDict(((child.path, child.populate(request, parent_path=active_path, alias=alias))
                                    for child in self.children))
        else:
            children = []

        show = (self.permissions and any([request.user.has_perm(perm) for perm in self.permissions])) or not \
            self.permissions

        return dict(self.kwargs, **{
            'title': self.title,
            'path': active_path,
            'url_name': self.url_name,
            'url': self.url,
            'children': children,
            'show': show,
            'active': active
        })


class BaseNavigation(object):
    """
    Base navigation
    """

    def __init__(self, request, alias=None):
        self.request = request
        self.alias = alias

    @abc.abstractmethod
    def init_nodes(self):
        pass

    def _populate_nodes(self, nodes):
        return OrderedDict(((node.path, node.populate(self.request, alias=self.alias)) for node in nodes))

    def get_nodes(self):
        return self._populate_nodes(self.init_nodes())


def get_nodes(request, alias):
    if hasattr(request, settings.NAVIGATOR_ATTR):
        navigation_group = getattr(request, settings.NAVIGATOR_ATTR, {})
        keys = alias.split('.')
        last_group = navigation_group
        for i, k in enumerate(keys, start=1):
            if k in last_group and not 'children' in last_group[k]:
                last_group = last_group[k]
            elif k in last_group and 'children' in last_group[k]:
                last_group = last_group[k]['children']
            else:
                last_group = {}
        return last_group