# -*- coding: utf-8
"""Implements PageBaseContent model"""

from __future__ import unicode_literals

from django.db import models
from django.utils import timezone
from django.utils.translation import ugettext_lazy as _
from django.conf import settings as django_settings
from django.utils.encoding import python_2_unicode_compatible

from pages.conf import settings
from pages.models.pagecontenttype import get_all_content_tuple


@python_2_unicode_compatible
class PageBaseContent(models.Model):
    type = models.CharField(_('Type'), choices=get_all_content_tuple(), max_length=100, blank=False, db_index=True)
    page = models.ForeignKey('pages.Page', verbose_name=_('Page'))
    language = models.CharField(max_length=5, default=str(settings.PAGES_DEFAULT_LANGUAGE))
    name = models.CharField(max_length=200, blank=True, unique=True)
    is_extended = models.BooleanField(_('Extended?'), default=False)
    comment = models.CharField(max_length=250, blank=True)
    date_created = models.DateTimeField(_('Created'), default=timezone.now)
    date_updated = models.DateTimeField(_('Updated'), default=timezone.now)
    created_by = models.ForeignKey(
        django_settings.AUTH_USER_MODEL, related_name='%(app_label)s_%(class)s_creator', null=True
    )
    updated_by = models.ForeignKey(
        django_settings.AUTH_USER_MODEL, related_name='%(app_label)s_%(class)s_editor', null=True
    )

    def __str__(self):
        """id string of instance"""
        return '{0:>s}'.format(self.name)

    def count_objects(self, use_lang=True):
        if use_lang:
            count = self.__class__.objects.filter(page=self.page, language=self.language).count()
        else:
            count = self.__class__.objects.filter(page=self.page).count()
        return count

    def update_fields(self, change):
        pass

    def save(self, force_insert=False, force_update=False, using=None, update_fields=None):
        """Override the default ``save`` method."""

        if self.pk is None:
            self.created_by = self.page.created_by
            self.update_fields(False)
        else:
            self.update_fields(True)

        count = self.count_objects()
        count = count - 1 if count else count
        self.name = '{0:>s}:{1:>s}:{2:>s}:{3:>d}'.format(
            self.language, self.type, self.page.name, count
        )

        self.updated_by = self.page.updated_by

        # Call parent's ``save`` function
        super(PageBaseContent, self).save(force_insert, force_update, using, update_fields)

    class Meta:
        abstract = True

    class PageMeta:
        def __init__(self):
            self.context_name = 'base'
            self.multiple_per_page = True
            self.multiple_per_locale = True
