#!/usr/bin/env python
# -*- coding: utf-8 -*-
from django.db import models
from django.contrib.auth.models import User

from datetime import datetime

from fitbit import Fitbit


class PhitbitUser(models.Model):

    def __init__(self, *args, **kwargs):
        c_id = kwargs.get('consumer_id', None)
        super(PhitbitUser, self).__init__(*args, **kwargs)
        if self._authenticated():
            consumer = PhitbitConsumer.objects.defaults(c_id)
            self.api = Fitbit(
                consumer_key=consumer.CONSUMER_KEY,
                consumer_secret=consumer.CONSUMER_SECRET,
                user_key=self.USER_KEY,
                user_secret=self.USER_SECRET
            )

    def __unicode__(self):
        return self.user.username

    def _authenticated(self):
        """ Simple authentication wrapper checks user keys."""

        if self.USER_KEY is not None and self.USER_SECRET is not None:
            self.authenticated = True
            self.save()
            return True
        self.authenticated = False
        self.save()
        return False

    def _build_oauth(self, c_id=None):
        """ Build and return the Fitbit OAuthclient

        Used during the login / complete views
        """
        consumer = PhitbitConsumer.objects.defaults(c_id)
        self.api = Fitbit(
            consumer_key=consumer.CONSUMER_KEY,
            consumer_secret=consumer.CONSUMER_SECRET,
        )
        return self.api

    def store(self, key, data):
        """ Store data in a PhitbitData object

        If the data already exists, saves the new data

        Parameters:
        name - desired name of the data to store
        data - what ever data you wish to store


        Returns the created PhitbitData object
        """

        phb, _ = PhitbitData.objects.get_or_create(
            phitbit_user=self,
            name=unicode(key),
            modified_date=datetime.now()
        )

        phb.data = data

        phb.save()

        return phb

    def retrieve(self, key):
        """ Retrieve a PhitbitData object

        Parameters:
        name - name of the data you wish to retrieve

        Returns None if object doesn't exist
        """

        phb = PhitbitData.objects.filter(
            phitbit_user=self,
            name=unicode(key)
        )

        if phb is not None:
            return phb[0]
        return None

    def goals(self):
        """ A wrapper around the Python-fitbit module returning goals """
        return self.api.activities()['goals']

    def summary(self):
        """ A wrapper around the Python-fitbit module returning the summary """
        return self.api.activities()['summary']

    def devices(self):
        """ Wrapper around Fitbit devices """
        return self.api.get_devices()

    def user_profile(self, user_id=None):
        """ Wrapper around Fitbit user_profile()

        If no user_id supplied - returns the current user
        """
        return self.api.user_profile_get(user_id=user_id)

    def _call_time_series(self, resource, period='1d', start=None, end=None):
        """ A helper method to call time_series """

        if start is None:
            data = self.api.time_series(
                resource,
                period=period,
            )
        else:
            data = self.api.time_series(
                resource,
                period=period,
                base_date=start,
                end_date=end
            )
        return data[resource.replace('/', '-')]

    def steps(self, period='1d', start=None, end=None):
        """ A wrapper around the fitbit api for 'activities/steps'

        Default is to return steps today

        Period must be '1d', '7d', '30d', '1w', '1m', '3m', '6m', '1y', 'max'
        """
        return self._call_time_series(
            'activities/steps', period, start, end)

    def distance(self, period='1d', start=None, end=None):
        """ A wrapper around the fitbit api for 'activities/distance'

        Period must be '1d', '7d', '30d', '1w', '1m', '3m', '6m', '1y', 'max'
        """
        return self._call_time_series(
            'activities/distance', period, start, end)

    def food_calories_in(self, period='1d', start=None, end=None):
        """ A wrapper around the fitbit api for 'activities/distance'

        Period must be '1d', '7d', '30d', '1w', '1m', '3m', '6m', '1y', 'max'
        """
        return self._call_time_series(
            'foods/logs/caloriesIn', period, start, end)

    def minutes_asleep(self, period='1d', start=None, end=None):
        """ A wrapper around the fitbit api for 'sleep/minutesAsleep'

        Period must be '1d', '7d', '30d', '1w', '1m', '3m', '6m', '1y', 'max'
        """
        return self._call_time_series(
            'sleep/minutesAsleep', period, start, end)

    def minutes_awake(self, period='1d', start=None, end=None):
        """ A wrapper around the fitbit api for 'sleep/minutesAwake'

        Period must be '1d', '7d', '30d', '1w', '1m', '3m', '6m', '1y', 'max'
        """
        return self._call_time_series(
            'sleep/minutesAwake', period, start, end)

    def sleep_efficiency(self, period='1d', start=None, end=None):
        """ A wrapper around the fitbit api for 'sleep/minutesAwake'

        Period must be '1d', '7d', '30d', '1w', '1m', '3m', '6m', '1y', 'max'
        """
        return self._call_time_series(
            'sleep/efficiency', period, start, end)

    USER_KEY = models.CharField(max_length=100, blank=True)

    USER_SECRET = models.CharField(max_length=100, blank=True)

    user = models.OneToOneField(User, related_name='phitbit')

    authenticated = models.BooleanField(default=False)


class PhitbitConsumerManager(models.Manager):

    def defaults(self, c_id=None):
        """ Returns the PhitbitConsumer object

        Is None is passed, return the first instance.

        You should always create your desired default option first.
        """
        if c_id is None:
            return PhitbitConsumer.objects.all()[0]
        else:
            return PhitbitConsumer.objects.get(pk=c_id)


class PhitbitConsumer(models.Model):

    objects = PhitbitConsumerManager()

    def __unicode__(self):
        return self.name

    name = models.CharField(max_length=50)

    CONSUMER_KEY = models.CharField(max_length=100, blank=True)

    CONSUMER_SECRET = models.CharField(max_length=100, blank=True)


class PhitbitData(models.Model):

    def __unicode__(self):
        return ' '.join(
            [
                self.name,
                'for user ',
                self.phitbit_user.user.username
            ]
        )

    phitbit_user = models.ForeignKey(PhitbitUser)

    data = models.CharField(max_length=100, blank=True)

    name = models.CharField(max_length=100)

    modified_date = models.DateTimeField(auto_now=True)
